package io.embrace.android.embracesdk

import android.content.Context
import android.os.Build
import android.os.PowerManager
import androidx.annotation.RequiresApi
import com.google.gson.annotations.SerializedName
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import java.io.Closeable
import java.util.LinkedList
import java.util.concurrent.Executor

private const val CAPTURE_LIMIT = 100

internal data class ThermalState(

    @SerializedName("t")
    internal val timestamp: Long,

    @SerializedName("s")
    internal val status: Int
)

@RequiresApi(Build.VERSION_CODES.Q)
internal class EmbraceThermalStatusService(
    ctx: Context,
    executor: Executor,
    private val clock: Clock,
    private val logger: InternalEmbraceLogger
) : Closeable, PowerManager.OnThermalStatusChangedListener, SessionEndListener {

    private val pm: PowerManager?
    private val thermalStates = LinkedList<ThermalState>()

    init {
        pm = try {
            runCatching {
                ctx.getSystemService(Context.POWER_SERVICE) as PowerManager
            }.getOrNull()
        } catch (exc: Throwable) {
            logger.logError("Failed to get PowerManager", exc)
            null
        }

        pm?.let {
            logger.logDeveloper("ThermalStatusService", "Adding thermal status listener")
            it.addThermalStatusListener(executor, this)
        }
    }

    private fun handleThermalStateChange(status: Int?) {
        if (status == null) {
            logger.logDeveloper("ThermalStatusService", "Null thermal status, no-oping.")
            return
        }

        logger.logDeveloper("ThermalStatusService", "Thermal status change: $status")
        thermalStates.add(ThermalState(clock.now(), status))

        if (thermalStates.size > CAPTURE_LIMIT) {
            logger.logDeveloper(
                "ThermalStatusService",
                "Exceeded capture limit, removing oldest thermal status sample."
            )
            thermalStates.removeFirst()
        }
    }

    override fun onThermalStatusChanged(status: Int) = handleThermalStateChange(status)

    override fun onSessionEnd(builder: Session.Builder) {
        builder.withBetaFeatures {
            it.thermalStates = thermalStates.toList()
        }
    }

    override fun close() {
        pm?.let {
            logger.logDeveloper("ThermalStatusService", "Removing thermal status listener")
            it.removeThermalStatusListener(this)
        }
    }
}
