package io.embrace.android.embracesdk;

import androidx.annotation.Nullable;

import java.lang.Thread.UncaughtExceptionHandler;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.Preconditions;

/**
 * Intercepts uncaught exceptions from the JVM and forwards them to the Embrace API. Once handled,
 * the exception is then delegated to the default {@link UncaughtExceptionHandler}.
 */
final class EmbraceUncaughtExceptionHandler implements UncaughtExceptionHandler {
    /** The default uncaught exception handler; is null if not set. */
    @Nullable
    private final UncaughtExceptionHandler defaultHandler;

    /** The crash service to submit the crashes to. */
    private final CrashService crashService;

    /**
     * Creates an instance of the {@link EmbraceUncaughtExceptionHandler}.
     *
     * @param defaultHandler the default {@link UncaughtExceptionHandler}, can be null
     * @param crashService the crash service which will submit the exception to the API as a crash
     */
    EmbraceUncaughtExceptionHandler(
            UncaughtExceptionHandler defaultHandler,
            CrashService crashService) {

        this.defaultHandler = defaultHandler;
        this.crashService = Preconditions.checkNotNull(crashService);
        InternalStaticEmbraceLogger.logDebug("Registered EmbraceUncaughtExceptionHandler");
    }

    @Override
    public void uncaughtException(Thread thread, Throwable exception) {
        try {
            crashService.handleCrash(thread, exception);
        } catch (Exception ex) {
            InternalStaticEmbraceLogger.logDebug("Error occurred in the uncaught exception handler", ex);
        } finally {
            InternalStaticEmbraceLogger.logDebug("Finished handling exception. Delegating to default handler.", exception);
            if (this.defaultHandler != null) {
                this.defaultHandler.uncaughtException(thread, exception);
            }
        }
    }
}
