package io.embrace.android.embracesdk

import android.os.Handler
import android.os.Looper
import androidx.annotation.VisibleForTesting
import io.embrace.android.embracesdk.config.AnrConfig
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger
import io.embrace.android.embracesdk.utils.NativeUtils
import java.util.*

private class UnityThreadSamplerNdkDelegate : EmbraceUnityThreadSamplerService.NdkDelegate {
    external override fun install(is32Bit: Boolean): Boolean
    external override fun sampleNdkStacktrace(): Int
    external override fun prepareNdkStacktraceSampling(ordinal: Int)
    external override fun fetchSample(): NativeStacktraceSample?
}

internal class UnityThreadSamplerInstaller(
    logger: InternalEmbraceLogger = InternalStaticEmbraceLogger.logger
) {

    private var installed = false
    private val unityHandler: Handler?

    init {
        // in Unity this should always run on the Unity thread.
        // We create a Handler here so that when the functionality is disabled locally
        // but enabled remotely, the config change callback also runs the install
        // on the Unity thread.
        if (Looper.myLooper() == null) {
            Looper.prepare()
        }

        val looper = Looper.myLooper()
        unityHandler = when {
            looper != null -> Handler(looper)
            else -> null
        }
        if (unityHandler == null) {
            logger.logError("Failed to create UnityThread Handler")
        }
    }

    fun install(
        sampler: UnityThreadSamplerService,
        configService: ConfigService,
        anrService: AnrService
    ) {
        if (configService.isUnityNdkSamplingEnabled) {
            install(sampler, anrService)
        } else {
            InternalStaticEmbraceLogger.logDeveloper(
                "UnityThreadSamplerInstaller",
                "isUnityNdkSamplingEnabled disabled."
            )
            // always install the handler. if config subsequently changes we take the decision
            // to just ignore anr intervals, rather than attempting to uninstall the handler
            configService.addListener { _: Config, _ ->
                onConfigChange(configService, sampler, anrService)
            }
        }
    }

    private fun onConfigChange(
        configService: ConfigService,
        sampler: UnityThreadSamplerService,
        anrService: AnrService
    ) {
        unityHandler?.post(
            Runnable {
                if (configService.isUnityNdkSamplingEnabled && !installed) {
                    InternalStaticEmbraceLogger.logDeveloper(
                        "UnityThreadSamplerInstaller",
                        "Unity NDK Sampling Enabled, proceed to install"
                    )
                    install(sampler, anrService)
                }
            }
        )
    }

    private fun install(sampler: UnityThreadSamplerService, anrService: AnrService) {
        synchronized(this) {
            if (!installed) {
                if (sampler.installNativeSampler()) {
                    InternalStaticEmbraceLogger.logDeveloper(
                        "UnityThreadSamplerInstaller",
                        "Native sampler installed"
                    )
                    installed = true
                    anrService.addBlockedThreadListener(sampler)
                }
            } else {
                InternalStaticEmbraceLogger.logDeveloper(
                    "UnityThreadSamplerInstaller",
                    "UnityThreadSamplerService already installed"
                )
            }
        }
    }
}

/**
 * Samples the UnityMain thread stacktrace when the thread is detected as blocked.
 *
 * The NDK layer must be enabled in order to use this functionality as this class
 * calls native code.
 */
internal class EmbraceUnityThreadSamplerService @JvmOverloads constructor(
    private val configService: ConfigService,
    private val clock: Clock,
    private val symbols: kotlin.Lazy<Map<String, String>?>,
    private val random: Random = Random(),
    private val logger: InternalEmbraceLogger = InternalStaticEmbraceLogger.logger,
    private val delegate: NdkDelegate = UnityThreadSamplerNdkDelegate()
) : UnityThreadSamplerService {

    internal interface NdkDelegate {
        fun install(is32Bit: Boolean): Boolean
        fun sampleNdkStacktrace(): Int
        fun prepareNdkStacktraceSampling(ordinal: Int)
        fun fetchSample(): NativeStacktraceSample?
    }

    companion object {
        private const val UNITY_THREAD_NAME = "UnityMain"
        private const val MAX_SAMPLE_POLL_MS = 100L
        private const val SAMPLE_POLL_INTERVAL_MS = 1L
    }

    @VisibleForTesting
    internal var ignored = true

    @VisibleForTesting
    internal var count = -1

    @VisibleForTesting
    internal var factor = -1

    @VisibleForTesting
    internal var samples: MutableList<NativeThreadSample> = mutableListOf()

    @VisibleForTesting
    internal val currentSample: NativeThreadSample?
        get() = samples.lastOrNull()

    private val unityThread: Thread = Thread.currentThread()

    override fun installNativeSampler(): Boolean {
        logger.logDeveloper("EmbraceUnityThreadSamplerService", "installNativeSampler")
        return if (isUnityMainThread()) {
            logger.logDeveloper(
                "EmbraceUnityThreadSamplerService",
                "Unity main thread, attempting to install UnityThreadSampler"
            )
            delegate.install(NativeUtils.is32BitDevice())
        } else {
            logger.logWarning("Attempted to install UnityThreadSampler but UnityMain not found. Skipping setup.")
            logger.logDeveloper(
                "EmbraceUnityThreadSamplerService",
                "UnityThreadSampler skipping setup"
            )
            false
        }
    }

    @VisibleForTesting
    internal fun isUnityMainThread() = Thread.currentThread().name == UNITY_THREAD_NAME

    override fun onThreadBlocked(thread: Thread, timestamp: Long) {
        logger.logDeveloper("EmbraceUnityThreadSamplerService", "onThreadBlocked")

        // use consistent config for the duration of this ANR interval.
        val anrConfig = configService.config.anrConfig
        ignored = !containsUnityStackframes(anrConfig, unityThread.stackTrace)
        if (ignored || shouldSkipNewSample(anrConfig)) {
            // we've reached the data capture limit - ignore any thread blocked intervals.
            logger.logDeveloper(
                "UnityThreadSamplerInstaller",
                "Data capture limit reached. Ignoring thread blocked intervals."
            )
            ignored = true
            return
        }

        val unwinder = anrConfig.getUnityNdkSamplingUnwinder()
        delegate.prepareNdkStacktraceSampling(unwinder.code)
        factor = anrConfig.getUnityNdkSamplingFactor()
        val offset = random.nextInt(factor)
        count = (factor - offset) % factor

        logger.logDeveloper("EmbraceUnityThreadSamplerService", "add NativeThreadSample samples")
        samples.add(
            NativeThreadSample(
                unityThread.id,
                unityThread.name,
                unityThread.priority,
                offset * anrConfig.getIntervalMs(),
                timestamp,
                mutableListOf(),
                mapThreadState(unityThread.state),
                unwinder
            )
        )
    }

    override fun onThreadBlockedInterval(thread: Thread, timestamp: Long) {
        logger.logDeveloper("EmbraceUnityThreadSamplerService", "onThreadBlockedInterval")

        if (ignored || !configService.isUnityNdkSamplingEnabled) {
            logger.logDeveloper(
                "UnityThreadSamplerInstaller",
                "Ignoring thread blocked interval"
            )
            return
        }
        if (count % factor == 0) {
            count = 0

            // trigger a sample of the thread then poll for the result...
            logger.logDeveloper(
                "EmbraceUnityThreadSamplerService",
                "trigger a sample of the thread"
            )
            val result = delegate.sampleNdkStacktrace()

            logger.logDeveloper(
                "EmbraceUnityThreadSamplerService",
                "sampleNdkStacktrace result: $result"
            )

            // retrieve the result, and record the failure code if the sample failed
            val sample = when {
                result != 0 -> {
                    logger.logDeveloper(
                        "EmbraceUnityThreadSamplerService",
                        "NativeStacktraceSample"
                    )
                    NativeStacktraceSample(result, null, null, null, null, null)
                }
                else -> {
                    logger.logDeveloper(
                        "EmbraceUnityThreadSamplerService",
                        "retrieveSample"
                    )
                    retrieveSample()
                }
            }

            // process the result and add a tick to the collection...
            sample?.run {
                sampleTimestamp = timestamp - (currentSample?.threadBlockedTimestamp ?: 0)
                sampleDurationMs = clock.now() - timestamp
                currentSample?.sample?.add(sample)

                logger.logDeveloper(
                    "UnityThreadSamplerInstaller",
                    "Added sample"
                )
            }
        }
        count++
    }

    override fun onThreadUnblocked(thread: Thread, timestamp: Long) {
        logger.logDeveloper(
            "UnityThreadSamplerInstaller",
            "Thread unblocked: ${thread.id}"
        )
        ignored = true
    }

    override fun cleanCollections() {
        logger.logDeveloper(
            "UnityThreadSamplerInstaller",
            "Clean collections"
        )
        samples = mutableListOf()
    }

    /**
     * Retrieves a sample by polling via the JNI until the Unity thread's signal handler
     * has completed. This function MUST only be called from the ANR monitor thread, as it
     * caches JNI references.
     *
     * In practice unwinding usually takes ~1ms with libunwind so this shouldn't
     * block for long - a maximum wait time of [MAX_SAMPLE_POLL_MS] serves as a timeout
     * to prevent the monitor thread from being blocked for too long.
     */
    private fun retrieveSample(): NativeStacktraceSample? {
        var obj: NativeStacktraceSample? = null
        var remainingMs = MAX_SAMPLE_POLL_MS

        while (remainingMs > 0 && obj == null) {
            Thread.sleep(SAMPLE_POLL_INTERVAL_MS)
            remainingMs -= SAMPLE_POLL_INTERVAL_MS
            obj = delegate.fetchSample()
        }
        return obj
    }

    private fun shouldSkipNewSample(anrConfig: AnrConfig): Boolean {
        val sessionLimit = anrConfig.getMaxAnrCapturedIntervalsPerSession()
        return !configService.isUnityNdkSamplingEnabled || samples.size >= sessionLimit
    }

    override fun onSessionEnd(builder: Session.Builder) {
        logger.logDeveloper(
            "EmbraceUnityThreadSamplerService",
            "onSessionEnd"
        )
        if (!configService.isUnityNdkSamplingEnabled) {

            logger.logDeveloper(
                "EmbraceUnityThreadSamplerService",
                "Unity Ndk Sampling not enabled"
            )
            return
        }

        logger.logDeveloper(
            "UnityThreadSamplerInstaller",
            "Unity NDK sampling enabled"
        )

        // the ANR might end before samples with offsets are recorded - avoid
        // recording an empty sample in the payload if this is the case.
        val usefulSamples = samples.toList().filter { it.sample?.isNotEmpty() ?: false }

        if (usefulSamples.isEmpty()) {
            logger.logDeveloper(
                "UnityThreadSamplerInstaller",
                "UsefulSamples empty"
            )
            return
        }

        builder
            .withNativeSampleTicks(usefulSamples.toList())
            .withNativeSymbols(symbols.value)
    }

    /**
     * Determines whether or not we should sample the Unity thread based on the thread stacktrace
     * and the ANR config.
     */
    @VisibleForTesting
    internal fun containsUnityStackframes(
        anrConfig: AnrConfig,
        stacktrace: Array<StackTraceElement>
    ): Boolean {
        if (anrConfig.getIgnoreUnityNdkSamplingAllowlist()) {
            logger.logDeveloper(
                "UnityThreadSamplerInstaller",
                "Ignore unity NDK sampling allow list"
            )
            return true
        }
        val allowlist = anrConfig.getUnityNdkSamplingAllowlist()

        logger.logDeveloper(
            "EmbraceUnityThreadSamplerService",
            "containsUnityStackframes allowlist size: " + allowlist.size
        )

        return stacktrace.any { frame ->
            allowlist.any { allowed ->
                frame.methodName == allowed.method && frame.className == allowed.clz
            }
        }
    }
}
