package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.utils.optional.Optional;

import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

class EmbraceUserService implements ActivityListener, UserService, SessionEndListener {

    // Valid persona regex representation.
    static final Pattern VALID_PERSONA = Pattern.compile("^[a-zA-Z0-9_]{1,32}$");
    // Maximum number of allowed personas.
    static final int PERSONA_LIMIT = 10;

    private final PreferencesService preferencesService;
    private volatile UserInfo userInfo;

    private final InternalEmbraceLogger logger;

    EmbraceUserService(
            PreferencesService preferencesService,
            InternalEmbraceLogger logger) {

        this.preferencesService = preferencesService;
        this.userInfo = UserInfo.ofStored(preferencesService);
        this.logger = logger;
    }

    @Override
    public Optional<UserInfo> loadUserInfoFromDisk() {
        try {
            return Optional.fromNullable(UserInfo.ofStored(preferencesService));
        } catch (Exception ex) {
            logger.logDebug("Failed to load user info from persistent storage.");
            return Optional.absent();
        }
    }

    @Override
    public UserInfo getUserInfo() {
        return userInfo;
    }

    @Override
    public void setUserIdentifier(@Nullable String userId) {
        String currentUserId = userInfo.getUserId();
        if (currentUserId != null && currentUserId.equals(userId)) {
            return;
        }
        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        this.userInfo = builder
                .withUserId(userId)
                .build();
        preferencesService.setUserIdentifier(userId);
    }

    @Override
    public void clearUserIdentifier() {
        setUserIdentifier(null);
    }

    @Override
    public void setUsername(@Nullable String username) {
        String currentUserName = userInfo.getUsername();
        if (currentUserName != null && currentUserName.equals(username)) {
            return;
        }

        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        this.userInfo = builder
                .withUsername(username)
                .build();
        preferencesService.setUsername(username);
    }

    @Override
    public void clearUsername() {
        setUsername(null);
    }

    @Override
    public void setUserEmail(@Nullable String email) {
        String currentEmail = userInfo.getEmail();
        if (currentEmail != null && currentEmail.equals(email)) {
            return;
        }

        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        this.userInfo = builder
                .withEmail(email)
                .build();
        preferencesService.setUserEmailAddress(email);
    }

    @Override
    public void clearUserEmail() {
        setUserEmail(null);
    }

    @Override
    public void setUserAsPayer() {
        setUserPersona(UserInfo.PERSONA_PAYER);
    }

    @Override
    public void clearUserAsPayer() {
        clearUserPersona(UserInfo.PERSONA_PAYER);
    }

    @Override
    public void setUserPersona(@NonNull String persona) {
        if (!VALID_PERSONA.matcher(persona).matches()) {
            logger.logWarning("Ignoring persona " + persona + " as it does not match " + VALID_PERSONA.pattern());
            return;
        }
        Set<String> currentPersonas = userInfo.getPersonas();
        if (currentPersonas != null) {
            if (currentPersonas.size() >= PERSONA_LIMIT) {
                logger.logWarning("Cannot set persona as the limit of " + PERSONA_LIMIT + " has been reached");
                return;
            }

            if (currentPersonas.contains(persona)) {
                return;
            }
        }

        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (builder.getPersonas() != null) {
            personas.addAll(builder.getPersonas());
        }
        personas.add(persona);

        this.userInfo = builder
                .withPersonas(personas)
                .build();
        preferencesService.setUserPersonas(personas);
    }

    @Override
    public void clearUserPersona(@NonNull String persona) {
        Set<String> currentPersonas = userInfo.getPersonas();
        if (currentPersonas != null && !currentPersonas.contains(persona)) {
            logger.logWarning("Persona '" + persona + "' is not set");
            return;
        }

        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (builder.getPersonas() != null) {
            personas.addAll(builder.getPersonas());
        }
        personas.remove(persona);
        this.userInfo = builder
                .withPersonas(personas)
                .build();
        preferencesService.setUserPersonas(personas);
    }

    @Override
    public void clearAllUserPersonas() {
        Set<String> currentPersonas = userInfo.getPersonas();
        if (currentPersonas != null && currentPersonas.isEmpty()) {
            return;
        }

        UserInfo.Builder builder = UserInfo.newBuilder(userInfo);
        Set<String> personas = new HashSet<>();
        if (preferencesService.getUserPayer()) {
            personas.add(UserInfo.PERSONA_PAYER);
        }
        if (preferencesService.isUsersFirstDay()) {
            personas.add(UserInfo.PERSONA_FIRST_DAY_USER);
        }
        this.userInfo = builder
                .withPersonas(personas)
                .build();
        preferencesService.setUserPersonas(personas);
    }

    @Override
    public void onSessionEnd(@NonNull Session.Builder builder) {
        builder.withUserInfo(Optional.of(UserInfo.newBuilder(userInfo).build()));
    }

    @Override
    public void clearAllUserInfo() {
        clearUserIdentifier();
        clearUserEmail();
        clearUsername();
        clearAllUserPersonas();
    }
}
