package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.google.gson.annotations.SerializedName;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Immutable object which represents an event within the app.
 */
@InternalApi
public final class Event {
    @SerializedName("n")
    private final String name;

    /**
     * Unique identifier used to refer to a remote log item.
     */
    @SerializedName("li")
    private final String messageId;

    /**
     * Unique identifier used to refer to all other types of events.
     */
    @SerializedName("id")
    private final String eventId;

    @SerializedName("si")
    private final String sessionId;

    @SerializedName("t")
    private final EmbraceEvent.Type type;

    @SerializedName("ts")
    private final Long timestamp;

    @SerializedName("th")
    private final Long lateThreshold;

    @SerializedName("sc")
    @VisibleForTesting
    final boolean screenshotTaken;

    @SerializedName("du")
    private final Long duration;

    @SerializedName("st")
    private final String appState;

    // serialized name matches handled exception name on iOS
    @SerializedName("he")
    @VisibleForTesting
    final Boolean isException;

    @SerializedName("pr")
    private final Map<String, Object> customProperties;

    @SerializedName("sp")
    private final Map<String, String> sessionProperties;

    private final transient List<String> activeEventIds;

    Event(Builder builder) {
        name = builder.name;
        eventId = builder.eventId;
        messageId = builder.messageId;
        sessionId = builder.sessionId;
        type = builder.type;
        timestamp = builder.timestamp;
        lateThreshold = builder.lateThreshold;
        screenshotTaken = builder.screenshotTaken;
        duration = builder.duration;
        appState = builder.appState;
        customProperties = builder.customProperties;
        activeEventIds = builder.activeEventIds;
        isException = builder.isException;
        sessionProperties = builder.sessionProperties;
    }

    public String getName() {
        return name;
    }

    public String getEventId() {
        return eventId;
    }

    public String getMessageId() {
        return messageId;
    }

    public String getSessionId() {
        return sessionId;
    }

    public EmbraceEvent.Type getType() {
        return type;
    }

    public Long getTimestamp() {
        return timestamp;
    }

    public Long getLateThreshold() {
        return lateThreshold;
    }

    public boolean isScreenshotTaken() {
        return screenshotTaken;
    }

    public Long getDuration() {
        return duration;
    }

    public String getAppState() {
        return appState;
    }

    public Map<String, Object> getCustomProperties() {
        return customProperties;
    }

    public Map<String, String> getSessionProperties() {
        return sessionProperties;
    }

    public List<String> getActiveEventIds() {
        return activeEventIds;
    }

    public Boolean isException() {
        return isException;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(Event copy) {
        Builder eventBuilder = new Builder();

        eventBuilder.withAppState(copy.getAppState());
        eventBuilder.withName(copy.getName());
        eventBuilder.withEventId(copy.getEventId());
        eventBuilder.withSessionId(copy.getSessionId());
        eventBuilder.withTimestamp(copy.getTimestamp());
        eventBuilder.withSessionProperties(copy.getSessionProperties());
        eventBuilder.withType(copy.getType());
        eventBuilder.withCustomProperties(copy.getCustomProperties());
        eventBuilder.withScreenshotTaken(copy.isScreenshotTaken());
        eventBuilder.withActiveEventIds(copy.getActiveEventIds());
        eventBuilder.withDuration(copy.getDuration());
        eventBuilder.withIsException(copy.isException());
        eventBuilder.withLateThreshold(copy.getLateThreshold());
        eventBuilder.withMessageId(copy.getMessageId());

        return eventBuilder;
    }

    @InternalApi
    public static final class Builder {
        String name;
        String eventId = Uuid.getEmbUuid();
        String messageId;
        String sessionId;
        EmbraceEvent.Type type;
        Long timestamp = System.currentTimeMillis();
        Long lateThreshold;
        boolean screenshotTaken;
        Long duration;
        String appState;
        Boolean isException;
        Map<String, Object> customProperties;
        Map<String, String> sessionProperties;
        List<String> activeEventIds;

        Builder() {
        }

        @NonNull
        public Builder withName(String name) {
            this.name = name;
            return this;
        }

        @NonNull
        public Builder withEventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        @NonNull
        public Builder withMessageId(String messageId) {
            this.messageId = messageId;
            return this;
        }

        @NonNull
        public Builder withSessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        @NonNull
        public Builder withType(EmbraceEvent.Type type) {
            this.type = type;
            return this;
        }

        @NonNull
        public Builder withTimestamp(Long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        @NonNull
        public Builder withLateThreshold(Long lateThreshold) {
            this.lateThreshold = lateThreshold;
            return this;
        }

        @NonNull
        public Builder withScreenshotTaken(boolean screenshotTaken) {
            this.screenshotTaken = screenshotTaken;
            return this;
        }

        @NonNull
        public Builder withDuration(Long duration) {
            this.duration = duration;
            return this;
        }

        @NonNull
        public Builder withAppState(String appState) {
            this.appState = appState;
            return this;
        }

        @NonNull
        public Builder withIsException(Boolean isException) {
            this.isException = isException;
            return this;
        }

        @NonNull
        public Builder withCustomProperties(@Nullable Map<String, Object> customProperties) {
            this.customProperties = (customProperties != null) ? new HashMap<>(customProperties) : null;
            return this;
        }

        @NonNull
        public Builder withSessionProperties(@Nullable Map<String, String> sessionProperties) {
            this.sessionProperties = (sessionProperties != null) ? new HashMap<>(sessionProperties) : null;
            return this;
        }

        @NonNull
        public Builder withActiveEventIds(@Nullable List<String> activeEventIds) {
            this.activeEventIds = (activeEventIds != null) ? new ArrayList<>(activeEventIds) : null;
            return this;
        }

        @NonNull
        public Map<String, Object> getCustomProperties() {
            return customProperties;
        }

        @NonNull
        public Map<String, String> getSessionProperties() {
            return sessionProperties;
        }

        @NonNull
        public Event build() {
            return new Event(this);
        }
    }
}
