package io.embrace.android.embracesdk

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import java.util.concurrent.TimeUnit

/**
 * The time default period after which an event is considered 'late'.
 */
private const val DEFAULT_LATE_THRESHOLD_MILLIS = 5000L

/**
 * This class is in charge of building events and sending them to our servers.
 */
internal class EventHandler(
    private val metadataService: MetadataService,
    private val configService: ConfigService,
    private val userService: UserService,
    private val screenshotService: ScreenshotService,
    private val performanceInfoService: PerformanceInfoService,
    private val gatingService: GatingService,
    private val apiClient: ApiClient,
    private val logger: InternalEmbraceLogger,
    private val clock: Clock
) {

    companion object {
        @JvmStatic
        fun isStartupEvent(eventName: String) = eventName == EmbraceEventService.STARTUP_EVENT_NAME

        @JvmStatic
        fun getInternalEventKey(eventName: String, identifier: String?) =
            if (identifier.isNullOrEmpty()) eventName else "$eventName#$identifier"
    }

    /**
     * Posts a 'LATE' event when the timer expires.
     */
    private val lateEventWorker: ScheduledWorker =
        ScheduledWorker.ofSingleThread("Late Event Handler")

    /**
     * Responsible for handling the start of an event.
     */
    fun onEventStarted(
        eventId: String,
        eventName: String,
        startTime: Long,
        allowScreenshot: Boolean,
        sessionProperties: EmbraceSessionProperties,
        eventProperties: Map<String, Object>?,
        timeoutCallback: Runnable
    ): EventDescription {
        val threshold = calculateLateThreshold(eventId)
        val event = buildStartEvent(
            eventId,
            eventName,
            startTime,
            threshold,
            sessionProperties,
            eventProperties
        )

        val timer = lateEventWorker.scheduleWithDelay(
            timeoutCallback,
            threshold - calculateOffset(startTime, threshold),
            TimeUnit.MILLISECONDS
        )

        if (shouldSendMoment(eventName)) {
            val eventMessage = buildStartEventMessage(event)
            apiClient.sendEvent(eventMessage)
        } else {
            logger.logDebug("$eventName start moment not sent based on gating config.")
        }

        return EventDescription(timer, event, allowScreenshot)
    }

    /**
     * Responsible for handling ending an event.
     *
     * @return the event message for the end event
     */
    fun onEventEnded(
        originEventDescription: EventDescription,
        late: Boolean,
        eventProperties: Map<String, Object>?,
        sessionProperties: EmbraceSessionProperties
    ): EventMessage {

        val event: Event = originEventDescription.event
        val startTime = event.timestamp
        val endTime = clock.now()
        val duration = Math.max(0, endTime - startTime)
        val screenshotTaken = handleScreenshot(late, originEventDescription)
        // cancel late scheduler
        originEventDescription.lateTimer.cancel(false)

        val endEvent = buildEndEvent(
            event,
            endTime,
            duration,
            screenshotTaken,
            late,
            sessionProperties,
            eventProperties
        )
        val endEventMessage = buildEndEventMessage(endEvent, startTime, endTime)

        if (shouldSendMoment(event.name)) {
            apiClient.sendEvent(endEventMessage)
        } else {
            logger.logDebug("${event.name} end moment not sent based on gating config.")
        }

        return endEventMessage
    }

    fun onClose() {
        lateEventWorker.close()
    }

    /**
     * It determines if given event is allowed to be started.
     */
    fun isAllowedToStart(eventName: String): Boolean {
        return if (eventName.isNullOrEmpty()) {
            logger.logWarning("Event name is empty. Ignoring this event.")
            false
        } else if (configService.isEventDisabled(eventName)) {
            logger.logWarning("Event disabled. Ignoring event with name $eventName")
            false
        } else if (configService.isMessageTypeDisabled(MessageType.EVENT)) {
            logger.logWarning("Event message disabled. Ignoring all Events.")
            false
        } else if (lateEventWorker.isShutdown) {
            logger.logError("Cannot start event as service is shut down")
            false
        } else true
    }

    /**
     * It determines if given event is allowed to be ended.
     */
    fun isAllowedToEnd(eventName: String): Boolean {
        return if (configService.isMessageTypeDisabled(MessageType.EVENT)) {
            logger.logWarning("Event message disabled. Ignoring all Events.")
            false
        } else true
    }

    fun buildStartupEventInfo(originEvent: Event, endEvent: Event): StartupEventInfo =
        StartupEventInfo.newBuilder()
            .withDuration(endEvent.duration)
            .withThreshold(originEvent.lateThreshold)
            .build()

    private inline fun buildEndEventMessage(event: Event, startTime: Long, endTime: Long) =
        EventMessage.newBuilder()
            .withPerformanceInfo(performanceInfoService.getPerformanceInfo(startTime, endTime))
            .withUserInfo(userService.userInfo)
            .withEvent(event)
            .build()

    /**
     * Checks if the moment (startup moment or a regular moment) should not be sent based on the
     * gating config.
     *
     * @param name of the moment
     * @return true if should be gated
     */
    private fun shouldSendMoment(name: String): Boolean {
        return if (name == EmbraceEventService.STARTUP_EVENT_NAME) {
            !gatingService.shouldGateStartupMoment()
        } else {
            !gatingService.shouldGateMoment()
        }
    }

    private inline fun buildStartEventMessage(event: Event) =
        EventMessage.newBuilder()
            .withUserInfo(userService.userInfo)
            .withAppInfo(metadataService.appInfo)
            .withDeviceInfo(metadataService.deviceInfo)
            .withEvent(event)
            .build()

    private fun buildStartEvent(
        eventId: String,
        eventName: String,
        startTime: Long,
        threshold: Long,
        sessionProperties: EmbraceSessionProperties,
        eventProperties: Map<String, Object>?
    ): Event {
        var builder = Event.newBuilder()
            .withEventId(eventId)
            .withType(EmbraceEvent.Type.START)
            .withAppState(metadataService.appState)
            .withName(eventName)
            .withLateThreshold(threshold)
            .withTimestamp(startTime)
            .withSessionProperties(sessionProperties.get())

        if (metadataService.activeSessionId.isPresent) {
            builder = builder.withSessionId(metadataService.activeSessionId.get())
        }

        eventProperties?.let {
            builder = builder.withCustomProperties(it)
        }

        return builder.build()
    }

    private fun buildEndEvent(
        originEvent: Event,
        endTime: Long,
        duration: Long,
        screenshotTaken: Boolean,
        late: Boolean,
        sessionProperties: EmbraceSessionProperties,
        eventProperties: Map<String, Object>?
    ): Event {

        val builder = Event.newBuilder()
            .withEventId(originEvent.eventId)
            .withAppState(metadataService.appState)
            .withTimestamp(endTime)
            .withDuration(duration)
            .withName(originEvent.name)
            .withScreenshotTaken(screenshotTaken)
            .withCustomProperties(eventProperties)
            .withSessionProperties(sessionProperties.get())
            .withType(if (late) EmbraceEvent.Type.LATE else EmbraceEvent.Type.END)
        if (metadataService.activeSessionId.isPresent) {
            builder.withSessionId(metadataService.activeSessionId.get())
        }

        return builder.build()
    }

    private fun calculateOffset(startTime: Long, threshold: Long): Long {
        // Ensure we adjust the threshold to take into account backdated events
        return Math.min(threshold, Math.max(0, clock.now() - startTime))
    }

    private fun calculateLateThreshold(eventId: String): Long {
        // Check whether a late threshold has been configured, otherwise use the default
        val limits = configService.config.eventLimits

        val value = limits[eventId]

        return when {
            value == null || !limits.containsKey(eventId) -> DEFAULT_LATE_THRESHOLD_MILLIS
            else -> value
        }
    }

    /**
     * It takes care of taking screenshot if necessary.
     *
     * @return true if screenshot has been taken
     */
    private fun handleScreenshot(late: Boolean, eventDescription: EventDescription): Boolean {
        if (shouldTakeScreenshot(late, eventDescription)) {
            return try {
                screenshotService.takeScreenshotMoment(eventDescription.event.eventId)
            } catch (ex: Exception) {
                logger.logWarning(
                    "Failed to take screenshot for event ${eventDescription.event.name}",
                    ex
                )
                false
            }
        }

        return false
    }

    private inline fun shouldTakeScreenshot(late: Boolean, eventDescription: EventDescription) =
        late && eventDescription.isAllowScreenshot && !configService.isScreenshotDisabledForEvent(
            eventDescription.event.name
        )
}
