package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.annotations.SerializedName;

/**
 * Represents an {@link Event} with metadata describing the device and user.
 */
@InternalApi
public final class EventMessage {

    @SerializedName("et")
    private final Event event;

    @SerializedName("cr")
    private final Crash crash;

    @SerializedName("d")
    private final DeviceInfo deviceInfo;

    @SerializedName("a")
    private final AppInfo appInfo;

    @SerializedName("u")
    private final UserInfo userInfo;

    @SerializedName("p")
    private final PerformanceInfo performanceInfo;

    @SerializedName("sk")
    private final Stacktraces stacktraces;

    @SerializedName("v")
    private final int version = ApiClient.MESSAGE_VERSION;

    @SerializedName("crn")
    private final NativeCrash nativeCrash;

    EventMessage(Builder builder) {
        event = builder.event;
        crash = builder.crash;
        deviceInfo = builder.deviceInfo;
        appInfo = builder.appInfo;
        userInfo = builder.userInfo;
        performanceInfo = builder.performanceInfo;
        stacktraces = builder.stacktraces;
        nativeCrash = builder.nativeCrash;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(EventMessage copy) {
        Builder messageBuilder = new Builder();

        messageBuilder.withEvent(copy.getEvent());
        messageBuilder.withAppInfo(copy.getAppInfo());
        messageBuilder.withCrash(copy.getCrash());
        messageBuilder.withUserInfo(copy.getUserInfo());
        messageBuilder.withDeviceInfo(copy.getDeviceInfo());
        messageBuilder.withPerformanceInfo(copy.getPerformanceInfo());
        messageBuilder.withStacktraces(copy.getStacktraces());
        //TODO: add native crash copy and Unit test for this class

        return messageBuilder;
    }

    public Event getEvent() {
        return event;
    }

    public Crash getCrash() {
        return crash;
    }

    public DeviceInfo getDeviceInfo() {
        return deviceInfo;
    }

    public AppInfo getAppInfo() {
        return appInfo;
    }

    public UserInfo getUserInfo() {
        return userInfo;
    }

    @Nullable
    public PerformanceInfo getPerformanceInfo() {
        return performanceInfo;
    }

    public Stacktraces getStacktraces() {
        return stacktraces;
    }

    public NativeCrash getNativeCrash() {
        return nativeCrash;
    }

    @InternalApi
    public static final class Builder {
        Event event;
        Crash crash;
        DeviceInfo deviceInfo;
        AppInfo appInfo;
        UserInfo userInfo;
        PerformanceInfo performanceInfo;
        Stacktraces stacktraces;
        NativeCrash nativeCrash;

        Builder() {
        }

        @NonNull
        public Builder withEvent(@NonNull Event event) {
            this.event = event;
            return this;
        }

        @NonNull
        public Builder withCrash(@Nullable Crash crash) {
            this.crash = crash;
            return this;
        }

        @NonNull
        public Builder withDeviceInfo(@NonNull DeviceInfo deviceInfo) {
            this.deviceInfo = deviceInfo;
            return this;
        }

        @NonNull
        public Builder withAppInfo(@NonNull AppInfo appInfo) {
            this.appInfo = appInfo;
            return this;
        }

        @NonNull
        public Builder withUserInfo(@NonNull UserInfo userInfo) {
            this.userInfo = userInfo;
            return this;
        }

        @NonNull
        public Builder withPerformanceInfo(@Nullable PerformanceInfo performanceInfo) {
            this.performanceInfo = performanceInfo;
            return this;
        }

        @NonNull
        public Builder withStacktraces(@Nullable Stacktraces stacktraces) {
            this.stacktraces = stacktraces;
            return this;
        }

        @NonNull
        public Builder withNativeCrash(@Nullable NativeCrash nativeCrash) {
            this.nativeCrash = nativeCrash;
            return this;
        }

        @NonNull
        public EventMessage build() {
            return new EventMessage(this);
        }
    }
}
