package io.embrace.android.embracesdk;

import androidx.annotation.VisibleForTesting;

import com.google.gson.annotations.SerializedName;

import java.util.ArrayList;
import java.util.List;

/**
 * Describes an Exception Error with a count of occurrences and a list of exceptions (causes).
 */
final class ExceptionError {

    /**
     * The occurrences list limit.
     */
    private static final int DEFAULT_EXCEPTION_ERROR_LIMIT = 5;
    private static final int DEFAULT_EXCEPTION_ERROR_LIMIT_STRICT_MODE = 50;

    @SerializedName("c")
    @VisibleForTesting
    int occurrences;

    @SerializedName("rep")
    @VisibleForTesting
    List<ExceptionErrorInfo> exceptionErrors;

    private Boolean logStrictMode = false;

    ExceptionError(Boolean logStrictMode) {
        this.occurrences = 0;
        this.exceptionErrors = new ArrayList<>();
        this.logStrictMode = logStrictMode;
    }

    /**
     * Add a new exception error info if exceptionErrors's size is below 20.
     * For each exceptions, occurrences is incremented by 1.
     *
     * @param ex       fo the exception error.
     * @param appState (foreground or background).
     */
    void addException(Throwable ex, String appState, Clock clock) {
        occurrences++;

        int exceptionsLimits = DEFAULT_EXCEPTION_ERROR_LIMIT;
        if (logStrictMode) {
            exceptionsLimits = DEFAULT_EXCEPTION_ERROR_LIMIT_STRICT_MODE;
        }

        if (exceptionErrors.size() < exceptionsLimits) {
            exceptionErrors.add(new
                    ExceptionErrorInfo(
                    clock.now(),
                    appState,
                    exceptionInfo(ex)
            ));
        }
    }

    private List<ExceptionInfo> exceptionInfo(Throwable ex) {
        List<ExceptionInfo> result = new ArrayList<>();
        Throwable throwable = ex;
        while (throwable != null && !throwable.equals(throwable.getCause())) {
            ExceptionInfo exceptionInfo = ExceptionInfo.ofThrowable(throwable);
            result.add(0, exceptionInfo);
            throwable = throwable.getCause();
        }
        return result;
    }
}
