package io.embrace.android.embracesdk;

import static io.embrace.android.embracesdk.KeyValueWriter.ESCAPE_CHAR;
import static io.embrace.android.embracesdk.KeyValueWriter.EXPECTED_TOKEN_LEN;
import static io.embrace.android.embracesdk.KeyValueWriter.STRING_COLLECTION_TOKEN;
import static io.embrace.android.embracesdk.KeyValueWriter.TOKEN;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;

/**
 * Reads key value pairs from an input stream and transforms them into an arbitrary type.
 *
 * @param <T> the type which the caller defines a transform for.
 */
abstract class KeyValueReader<T> {

    protected final InternalEmbraceLogger logger;

    protected KeyValueReader(InternalEmbraceLogger logger) {
        this.logger = logger;
    }

    /**
     * Reads key value pairs from an input stream and transforms them into an arbitrary type.
     */
    @Nullable
    public final T read(@NonNull InputStream stream) {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(stream))) {
            return readImpl(reader);
        } catch (IOException exc) {
            if (!(exc instanceof FileNotFoundException)) {
                logger.logWarning("Key-value reader failed to read file.", exc);
            }
            return null;
        }
    }

    /**
     * Convenience method for reading a sanitized collection of strings.
     */
    @Nullable
    public final List<String> readStringCollection(String input) {
        if (input == null) {
            return null;
        } else {
            return Arrays.asList(input.split(STRING_COLLECTION_TOKEN));
        }
    }

    @Nullable
    public final Integer intFromString(String value) {
        if (value == null) {
            return null;
        } else {
            try {
                return Integer.parseInt(value);
            } catch (NumberFormatException exc) {
                return null;
            }
        }
    }

    @Nullable
    public final Long longFromString(String value) {
        if (value == null) {
            return null;
        } else {
            try {
                return Long.parseLong(value);
            } catch (NumberFormatException exc) {
                return null;
            }
        }
    }

    @Nullable
    public final Boolean booleanFromString(String value) {
        if (value == null) {
            return null;
        } else {
            return Boolean.parseBoolean(value);
        }
    }

    private T readImpl(BufferedReader reader) throws IOException {
        Map<String, String> data = readPropertiesMap(reader);
        if (data.isEmpty()) {
            return null;
        }

        // be forgiving of any user-error and simply return null,
        // as we'll fetch the config from the API instead.
        try {
            return transform(data);
        } catch (Exception exc) {
            logger.logWarning("Key-value reader failed to transform map to object.", exc);
            return null;
        }
    }

    private Map<String, String> readPropertiesMap(BufferedReader reader) throws IOException {
        String line;
        Map<String, String> data = new HashMap<>();

        while ((line = reader.readLine()) != null) {
            String[] tokens = line.split(TOKEN);

            if (tokens.length < EXPECTED_TOKEN_LEN) {
                throw new IllegalArgumentException("Read invalid key-value: " + line);
            }
            String key = tokens[0];
            String value = line.substring(key.length() + 1);
            data.put(key, unescapeValue(value));
        }
        return data;
    }

    private String unescapeValue(String value) {
        String sanitizedVal = value;
        sanitizedVal = unescapeString(sanitizedVal, TOKEN);
        sanitizedVal = unescapeString(sanitizedVal, STRING_COLLECTION_TOKEN);
        return sanitizedVal;
    }

    private String unescapeString(String value, String token) {
        if (value.contains(token)) {
            return value.replaceAll(ESCAPE_CHAR + token, token);
        }
        return value;
    }

    @Nullable
    abstract T transform(@Nullable Map<String, String> data) throws Exception;
}
