package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;

/**
 * Writes key value pairs from an arbitrary type to an OutputStream.
 *
 * @param <T> the type which the caller defines a transform for.
 */
abstract class KeyValueWriter<T> {

    static final String TOKEN = "=";
    static final String STRING_COLLECTION_TOKEN = ";";
    static final int EXPECTED_TOKEN_LEN = 2;
    static final String ESCAPE_CHAR = "\\\\";

    protected final InternalEmbraceLogger logger;

    protected KeyValueWriter(InternalEmbraceLogger logger) {
        this.logger = logger;
    }

    /**
     * Writes key value pairs from an arbitrary type to an OutputStream.
     */
    public final boolean write(@NonNull T data, @NonNull OutputStream stream) {
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(stream))) {
            return writeImpl(data, writer);
        } catch (IOException exc) {
            logger.logWarning("Key-value writer failed to read file.", exc);
            return false;
        }
    }

    /**
     * Convenience method for serializing a collection of strings, which will be delimited with
     * the ; token.
     */
    public final String serializeStringCollection(Collection<String> collection) {
        if (collection == null || collection.isEmpty()) {
            return null;
        } else {
            return StreamUtilsKt.join(STRING_COLLECTION_TOKEN, collection);
        }
    }

    private boolean writeImpl(T data, BufferedWriter writer) throws IOException {
        Map<String, String> props = sanitizeMap(transformImpl(data));

        if (props == null || props.isEmpty()) {
            return false;
        }

        for (Map.Entry<String, String> entry : props.entrySet()) {
            writer.write(entry.getKey() + TOKEN + entry.getValue());
            writer.newLine();
        }
        return true;
    }

    private Map<String, String> sanitizeMap(Map<String, String> props) {
        if (props == null) {
            return null;
        }
        Map<String, String> result = new HashMap<>();

        for (Map.Entry<String, String> entry : props.entrySet()) {
            String value = entry.getValue();

            if (value != null) {
                String sanitizedVal = value;
                sanitizedVal = escapeString(sanitizedVal, TOKEN);
                sanitizedVal = escapeString(sanitizedVal, STRING_COLLECTION_TOKEN);
                result.put(entry.getKey(), sanitizedVal);
            }
        }
        return result;
    }

    private String escapeString(String value, String token) {
        if (value.contains(token)) {
            return value.replaceAll(token, ESCAPE_CHAR + token);
        }
        return value;
    }

    private Map<String, String> transformImpl(T data) {
        try {
            return transform(data);
        } catch (Exception exc) {
            // be forgiving of any user-error and simply return null
            logger.logWarning("Key-value writer failed to transform object to map.", exc);
            return null;
        }
    }

    @Nullable
    abstract Map<String, String> transform(@NonNull T data) throws Exception;
}
