package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;

/**
 * Logs network calls made by the application. The Embrace SDK intercepts the calls and reports
 * them to the API.
 */
@InternalApi
public interface NetworkLoggingService {

    /**
     * Get the calls and counts of network calls (which exceed the limit) within the specified time
     * range.
     *
     * @param startTime     the start time
     * @param lastKnownTime the end time
     * @return
     */
    NetworkSessionV2 getNetworkCallsForSession(long startTime, long lastKnownTime);

    /**
     * Logs a HTTP network call.
     *
     * @param url           the URL being called
     * @param httpMethod    the HTTP method
     * @param statusCode    the status code from the response
     * @param startTime     the start time of the request
     * @param endTime       the end time of the request
     * @param bytesSent     the number of bytes sent
     * @param bytesReceived the number of bytes received
     * @param traceId optional trace ID that can be used to trace a particular request
     */
    void logNetworkCall(
            @NonNull String url,
            @NonNull String httpMethod,
            int statusCode,
            long startTime,
            long endTime,
            long bytesSent,
            long bytesReceived,
            String traceId);

    /**
     * Logs an exception which occurred when attempting to make a network call.
     *
     * @param url          the URL being called
     * @param httpMethod   the HTTP method
     * @param startTime    the start time of the request
     * @param endTime      the end time of the request
     * @param errorType    the type of error being thrown
     * @param errorMessage the error message
     * @param traceId optional trace ID that can be used to trace a particular request
     */
    void logNetworkError(
            @NonNull String url,
            @NonNull String httpMethod,
            long startTime,
            long endTime,
            String errorType,
            String errorMessage,
            String traceId);

}
