package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.google.gson.annotations.SerializedName;

import java.util.List;

/**
 * Describes information about how the device is performing.
 */
class PerformanceInfo {

    /** Current battery level of the device. */
    @SerializedName("bt")
    @VisibleForTesting
    final Float batteryLevel;

    /** Current disk space usage of the app, and free space on the device. */
    @SerializedName("ds")
    @VisibleForTesting
    final DiskUsage diskUsage;

    /** Memory usage at various points in time on the device. */
    @SerializedName("me")
    @VisibleForTesting
    final List<MemorySample> memorySamples;

    /** Occasions where the device reported that the memory is running low. */
    @SerializedName("mw")
    @VisibleForTesting
    final List<MemoryWarning> memoryWarnings;

    /** Periods during which the device was charging. */
    @SerializedName("bc")
    @VisibleForTesting
    final List<Interval> chargingIntervals;

    /** Periods during which the CPU usage was high. */
    @SerializedName("cp")
    private final List<Interval> criticalCpuIntervals;

    /** Periods during which the device was connected to WIFI, WAN, or no network. */
    @SerializedName("ns")
    @VisibleForTesting
    final List<Interval> networkInterfaceIntervals;

    /** Measurements of the battery level at various points in time. */
    @SerializedName("bm")
    @VisibleForTesting
    final List<BatteryMeasurement> batteryMeasurements;

    /** Periods during which the application was not responding (UI thread blocked for > 1 sec). */
    @SerializedName("anr")
    @VisibleForTesting
    final List<AnrInterval> anrIntervals;

    /** Timestamps where Google ANRs were triggered. */
    @SerializedName("ga")
    private final List<Long> googleAnrTimestamps;

    /** Measurements of signal strength on each change. */
    @SerializedName("ss")
    private final List<SignalStrength> signalStrength;

    /** Measurements of the connection quality at various points in time. */
    @SerializedName("cq")
    private final List<ConnectionQualityInterval> connectionQualityIntervals;

    protected PerformanceInfo(Builder builder) {
        batteryLevel = builder.batteryLevel;
        diskUsage = builder.diskUsage;
        memorySamples = builder.memorySamples;
        memoryWarnings = builder.memoryWarnings;
        chargingIntervals = builder.chargingIntervals;
        criticalCpuIntervals = builder.criticalCpuIntervals;
        networkInterfaceIntervals = builder.networkInterfaceIntervals;
        batteryMeasurements = builder.batteryMeasurements;
        anrIntervals = builder.anrIntervals;
        googleAnrTimestamps = builder.googleAnrTimestamps;
        signalStrength = builder.signalStrength;
        connectionQualityIntervals = builder.connectionQualityIntervals;
    }

    public Float getBatteryLevel() {
        return batteryLevel;
    }

    public DiskUsage getDiskUsage() {
        return diskUsage;
    }

    public List<MemorySample> getMemorySamples() {
        return memorySamples;
    }

    public List<MemoryWarning> getMemoryWarnings() {
        return memoryWarnings;
    }

    public List<Interval> getChargingIntervals() {
        return chargingIntervals;
    }

    public List<Interval> getCriticalCpuIntervals() {
        return criticalCpuIntervals;
    }

    public List<Interval> getNetworkInterfaceIntervals() {
        return networkInterfaceIntervals;
    }

    public List<BatteryMeasurement> getBatteryMeasurements() {
        return batteryMeasurements;
    }

    public List<AnrInterval> getAnrIntervals() {
        return anrIntervals;
    }

    public List<SignalStrength> getSignalStrength() {
        return signalStrength;
    }

    public List<ConnectionQualityInterval> getConnectionQualityIntervals() {
        return connectionQualityIntervals;
    }

    public static PerformanceInfo.Builder newBuilder() {
        return new PerformanceInfo.Builder();
    }

    public static PerformanceInfo.Builder newBuilder(@Nullable PerformanceInfo copy) {
        PerformanceInfo.Builder builder = new PerformanceInfo.Builder();

        if (copy == null) {
            return builder;
        }

        builder.withBatteryLevel(copy.getBatteryLevel());
        builder.withDiskUsage(copy.getDiskUsage());
        builder.withMemorySamples(copy.getMemorySamples());
        builder.withMemoryWarnings(copy.getMemoryWarnings());
        builder.withChargingIntervals(copy.getChargingIntervals());
        builder.withCriticalCpuIntervals(copy.getCriticalCpuIntervals());
        builder.withNetworkInterfaceIntervals(copy.getNetworkInterfaceIntervals());
        builder.withBatteryMeasurements(copy.getBatteryMeasurements());
        builder.withAnrIntervals(copy.getAnrIntervals());
        builder.withSignalStrength(copy.getSignalStrength());
        builder.withConnectionQualityIntervals(copy.getConnectionQualityIntervals());

        return builder;
    }

    static class Builder {
        Float batteryLevel;
        DiskUsage diskUsage;
        List<MemorySample> memorySamples;
        List<MemoryWarning> memoryWarnings;
        List<Interval> chargingIntervals;
        List<Interval> criticalCpuIntervals;
        List<Interval> networkInterfaceIntervals;
        List<BatteryMeasurement> batteryMeasurements;
        List<Long> googleAnrTimestamps;
        List<AnrInterval> anrIntervals;
        List<SignalStrength> signalStrength;
        List<ConnectionQualityInterval> connectionQualityIntervals;

        protected Builder() {
        }

        @NonNull
        public Builder withBatteryLevel(Float batteryLevel) {
            this.batteryLevel = batteryLevel;
            return this;
        }

        @NonNull
        public Builder withDiskUsage(DiskUsage diskUsage) {
            this.diskUsage = diskUsage;
            return this;
        }

        @NonNull
        public Builder withMemorySamples(@Nullable List<MemorySample> memorySamples) {
            this.memorySamples = memorySamples;
            return this;
        }

        @NonNull
        public Builder withMemoryWarnings(@Nullable List<MemoryWarning> memoryWarnings) {
            this.memoryWarnings = memoryWarnings;
            return this;
        }

        @NonNull
        public Builder withChargingIntervals(@Nullable List<Interval> chargingIntervals) {
            this.chargingIntervals = chargingIntervals;
            return this;
        }

        @NonNull
        public Builder withCriticalCpuIntervals(@Nullable List<Interval> criticalCpuIntervals) {
            this.criticalCpuIntervals = criticalCpuIntervals;
            return this;
        }

        @NonNull
        public Builder withNetworkInterfaceIntervals(@Nullable List<Interval> networkInterfaceIntervals) {
            this.networkInterfaceIntervals = networkInterfaceIntervals;
            return this;
        }

        @NonNull
        public Builder withBatteryMeasurements(@Nullable List<BatteryMeasurement> batteryMeasurements) {
            this.batteryMeasurements = batteryMeasurements;
            return this;
        }

        @NonNull
        public Builder withAnrIntervals(@Nullable List<AnrInterval> anrIntervals) {
            this.anrIntervals = anrIntervals;
            return this;
        }

        public Builder withGoogleAnrTimestamps(List<Long> googleAnrTimestamps) {
            this.googleAnrTimestamps = googleAnrTimestamps;
            return this;
        }

        @NonNull
        public Builder withSignalStrength(@Nullable List<SignalStrength> signalStrength) {
            this.signalStrength = signalStrength;
            return this;
        }

        @NonNull
        public Builder withConnectionQualityIntervals(@Nullable List<ConnectionQualityInterval> connectionQualityIntervals) {
            this.connectionQualityIntervals = connectionQualityIntervals;
            return this;
        }

        @NonNull
        public PerformanceInfo build() {
            return new PerformanceInfo(this);
        }
    }
}
