package io.embrace.android.embracesdk

import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_ANR
import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_BATTERY_LEVEL
import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_BATTERY_MEASUREMENTS
import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_CHARGING_INTERVALS
import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_CONNECTIVITY
import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_CPU
import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_CURRENT_DISK_USAGE
import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_LOW_MEMORY
import io.embrace.android.embracesdk.config.GatingConfig.Companion.PERFORMANCE_MEMORY_SAMPLES
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger

internal class PerformanceInfoSanitizer(
    private val performanceInfo: PerformanceInfo?,
    private val enabledComponents: Set<String>
) :
    Sanitizable<PerformanceInfo> {
    override fun sanitize(): PerformanceInfo? {
        return performanceInfo?.let {
            val sanitizedPerformanceInfoBuilder = PerformanceInfo.newBuilder(performanceInfo)

            if (!shouldSendANRs()) {
                sanitizedPerformanceInfoBuilder.withAnrIntervals(null)
            }

            if (!shouldSendBatteryLevel()) {
                sanitizedPerformanceInfoBuilder.withBatteryLevel(null)
            }

            if (!shouldSendBatteryMeasurements()) {
                sanitizedPerformanceInfoBuilder.withBatteryMeasurements(null)
            }

            if (!shouldSendChargingIntervals()) {
                sanitizedPerformanceInfoBuilder.withChargingIntervals(null)
            }

            if (!shouldSendCriticalCPUIntervals()) {
                sanitizedPerformanceInfoBuilder.withCriticalCpuIntervals(null)
            }

            if (!shouldSendNetworkConnectivityIntervals()) {
                sanitizedPerformanceInfoBuilder.withNetworkInterfaceIntervals(null)
            }

            if (!shouldSendLowMemoryWarnings()) {
                sanitizedPerformanceInfoBuilder.withMemoryWarnings(null)
            }

            if (!shouldSendMemorySamples()) {
                sanitizedPerformanceInfoBuilder.withMemorySamples(null)
            }

            if (!shouldSendCurrentDiskUsage()) {
                sanitizedPerformanceInfoBuilder.withDiskUsage(null)
            }

            InternalStaticEmbraceLogger.logger.logDeveloper(
                "PerformanceInfoSanitizer",
                "sanitize"
            )

            return sanitizedPerformanceInfoBuilder.build()
        }
    }

    private fun shouldSendANRs() =
        enabledComponents.contains(PERFORMANCE_ANR)

    private fun shouldSendCurrentDiskUsage() =
        enabledComponents.contains(PERFORMANCE_CURRENT_DISK_USAGE)

    private fun shouldSendCriticalCPUIntervals() =
        enabledComponents.contains(PERFORMANCE_CPU)

    private fun shouldSendBatteryLevel() =
        enabledComponents.contains(PERFORMANCE_BATTERY_LEVEL)

    private fun shouldSendMemorySamples() =
        enabledComponents.contains(PERFORMANCE_MEMORY_SAMPLES)

    private fun shouldSendBatteryMeasurements() =
        enabledComponents.contains(PERFORMANCE_BATTERY_MEASUREMENTS)

    private fun shouldSendChargingIntervals() =
        enabledComponents.contains(PERFORMANCE_CHARGING_INTERVALS)

    private fun shouldSendNetworkConnectivityIntervals() =
        enabledComponents.contains(PERFORMANCE_CONNECTIVITY)

    private fun shouldSendLowMemoryWarnings() =
        enabledComponents.contains(PERFORMANCE_LOW_MEMORY)
}
