package io.embrace.android.embracesdk;

import android.text.TextUtils;

import androidx.annotation.VisibleForTesting;

import java.util.HashMap;
import java.util.Map;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;

/**
 * This class is responsible for tracking app performance during purchase flows.
 * <p>
 * This class is thread-safe.
 */
public final class PurchaseFlow extends CustomFlow {
    static final String MOMENT_ADD_TO_CART = "_add-to-cart";
    static final String MOMENT_PURCHASE = "_purchase";

    static final String PROP_AMOUNT = "amount";
    static final String PROP_ITEM_ID = "item-id";
    static final String PROP_NUM_ITEMS = "num-items";
    static final String PROP_ORDER_ID = "order-id";
    static final String PROP_PAYMENT_TYPE = "payment-type";
    static final String PROP_PRICE = "price";
    static final String PROP_QUANTITY = "quantity";

    private final Map<String, Map<String, Object>> addToCartPropsMap = new HashMap<>();

    @VisibleForTesting
    Map<String, Map<String, Object>> getCart() {
        return addToCartPropsMap;
    }

    private Map<String, Object> purchaseProps = new HashMap<>();
    private volatile String purchaseMomentId;

    @VisibleForTesting
    String getPurchaseMomentId() {
        return purchaseMomentId;
    }

    /**
     * Starts an add-to-cart app moment.
     * <p>
     * This method should be called as soon as the user indicates an intent to add an item to their cart.
     *
     * @param itemId     The ID that represents the item being added to the cart. This value is optional and, if present, will
     *                   associate the value as a property of the moment.
     * @param quantity   The number of items being added to the cart. This value is optional and, if present, will associate
     *                   the value as a property of the moment.
     * @param price      The unit price of the item being added to the cart. This value is optional and, if present, will
     *                   associate the value as a property of the moment.
     * @param properties A map of Strings to Objects that represent additional properties to associate with the moment.
     *                   This value is optional. A maximum of 10 properties (not including the ones set via arguments to
     *                   this method) may be set.
     * @return A moment identifier that can be used to close the add-to-cart moment. If an error was encountered, this
     * method returns null.
     */
    public String addToCartStart(String itemId,
                                 Number quantity,
                                 Number price,
                                 Map<String, Object> properties) {

        String momentId = Uuid.getEmbUuid();

        Map<String, Object> normalizedProperties = PropertyUtils.sanitizeProperties(properties);

        if (itemId != null) {
            normalizedProperties.put(PROP_ITEM_ID, itemId);
        }
        if (quantity != null) {
            normalizedProperties.put(PROP_QUANTITY, quantity.toString());
        }
        if (price != null) {
            normalizedProperties.put(PROP_PRICE, price.toString());
        }

        sendMomentStartEvent(MOMENT_ADD_TO_CART, momentId, true, normalizedProperties);
        this.addToCartPropsMap.put(momentId, normalizedProperties);

        return momentId;
    }

    /**
     * Ends a particular add-to-cart moment instance and generates an info log message that indicates that adding to the
     * cart completed.
     * <p>
     * This method should be called once the item is verified to be in the user's cart.
     *
     * @param momentId The moment identifier returned by the `PurchaseFlow.addToCartStart` method. This moment identifier
     *                 must be an identifier produced by this particular PurchaseFlow instance and must not have already
     *                 been marked as completed or failed.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean addToCartComplete(String momentId) {
        return addToCartComplete(momentId, null);
    }

    /**
     * Ends a particular add-to-cart moment instance and generates an info log message that indicates that adding to the
     * cart completed.
     * <p>
     * This method should be called once the item is verified to be in the user's cart.
     *
     * @param momentId   The moment identifier returned by the `PurchaseFlow.addToCartStart` method. This moment identifier
     *                   must be an identifier produced by this particular PurchaseFlow instance and must not have already
     *                   been marked as completed or failed.
     * @param properties Custom properties to pass in as part of the add to cart completion
     * @return True if the operation was successful; false otherwise.
     */
    public boolean addToCartComplete(String momentId, Map<String, Object> properties) {
        if (TextUtils.isEmpty(momentId) || !this.addToCartPropsMap.containsKey(momentId)) {
            InternalStaticEmbraceLogger.logError("Purchase flow does not recognize add-to-cart moment identifier.");
            return false;
        }

        sendMomentEndEvent(MOMENT_ADD_TO_CART, momentId, properties);
        this.addToCartPropsMap.remove(momentId);

        return true;
    }


    /**
     * Ends a particular add-to-cart moment instance and generates an error log message that indicates that adding to the
     * cart failed.
     * <p>
     * This method should be called when it has been determined that the item could not be added to the cart.
     *
     * @param momentId   The moment identifier returned by the `PurchaseFlow.addToCartStart` method. This moment identifier
     *                   must be an identifier produced by this particular PurchaseFlow instance and must not have already
     *                   been marked as completed or failed.
     * @param msg        A message that explains the reason for why this operation failed. This value is optional and, if
     *                   provided, will associate the value as a property of the error log message.
     * @param properties Custom properties to pass in as part of the add to cart failure.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean addToCartFail(String momentId,
                                 String msg,
                                 Map<String, Object> properties) {
        if (TextUtils.isEmpty(momentId) || !this.addToCartPropsMap.containsKey(momentId)) {
            InternalStaticEmbraceLogger.logError("Purchase flow does not recognize add-to-cart moment identifier.");
            return false;
        }

        String errorLogMsg;
        if (msg != null) {
            this.addToCartPropsMap.get(momentId).put(PROP_MESSAGE, msg);
        }

        if (TextUtils.isEmpty(msg)) {
            errorLogMsg = "A failure occurred while adding an item to the cart.";
        } else {
            errorLogMsg = "A failure occurred while adding an item to the cart: " + msg;
        }

        sendLogError(errorLogMsg, true, this.addToCartPropsMap.get(momentId));
        sendMomentEndEvent(MOMENT_ADD_TO_CART, momentId, properties);

        this.addToCartPropsMap.remove(momentId);
        return true;
    }

    /**
     * Ends a particular add-to-cart moment instance and generates an error log message that indicates that adding to the
     * cart failed.
     * <p>
     * This method should be called when it has been determined that the item could not be added to the cart.
     *
     * @param momentId The moment identifier returned by the `PurchaseFlow.addToCartStart` method. This moment identifier
     *                 must be an identifier produced by this particular PurchaseFlow instance and must not have already
     *                 been marked as completed or failed.
     * @param msg      A message that explains the reason for why this operation failed. This value is optional and, if
     *                 provided, will associate the value as a property of the error log message.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean addToCartFail(String momentId,
                                 String msg) {

        return addToCartFail(momentId, msg, null);
    }

    /**
     * Starts a purchase moment.
     * <p>
     * This method should be called as soon as the user indicates an intent to purchase the items in their cart. This
     * means that all information pertaining to the order (e.g. billing, payment, shipping) should already be known prior
     * to invoking this method.
     *
     * @param orderId     The ID that represents the purchase order. This value is optional and, if present, will associate
     *                    the value as a property of the moment.
     * @param numItems    The number of items in the purchase order. This value is optional and, if present, will associate
     *                    the value as a property of the moment.
     * @param amount      The total amount of the purchase order. This value is optional and, if present, will associate the
     *                    value as a property of the moment.
     * @param paymentType The payment system that will be fulfilling the purchase order (e.g. Google IAB, PayPal,
     *                    BrainTree). This value is optional and, if present, will associate the value as a property of
     *                    the moment.
     * @param properties  A map of Strings to Objects that represent additional properties to
     *                    associate with the moment.
     *                    This value is optional. A maximum of 10 properties (not including the ones set via arguments to
     *                    this method) may be set.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean purchaseStart(String orderId,
                                 Number numItems,
                                 Number amount,
                                 String paymentType,
                                 Map<String, Object> properties) {


        Map<String, Object> normalizedProperties = PropertyUtils.sanitizeProperties(properties);

        if (orderId != null) {
            normalizedProperties.put(PROP_ORDER_ID, orderId);
        }
        if (numItems != null) {
            normalizedProperties.put(PROP_NUM_ITEMS, numItems.toString());
        }
        if (amount != null) {
            normalizedProperties.put(PROP_AMOUNT, amount.toString());
        }
        if (paymentType != null) {
            normalizedProperties.put(PROP_PAYMENT_TYPE, paymentType);
        }

        this.purchaseMomentId = Uuid.getEmbUuid();
        this.purchaseProps = normalizedProperties;

        sendMomentStartEvent(MOMENT_PURCHASE, this.purchaseMomentId, false, normalizedProperties);
        return true;
    }

    /**
     * Ends the purchase moment and generates an info log message that indicates that the purchase completed.
     * <p>
     * This method should be called once the purchase order has been confirmed.
     *
     * @param properties properties to pass in as part of the purchase completion
     * @return True if the operation was successful; false otherwise.
     */
    public boolean purchaseComplete(Map<String, Object> properties) {
        if (this.purchaseMomentId == null) {
            InternalStaticEmbraceLogger.logError("Purchase wasn't started.");
            return false;
        }

        sendMomentEndEvent(MOMENT_PURCHASE, this.purchaseMomentId, properties);
        sendLogInfo("Purchase was completed.", this.purchaseProps);

        // Since the user has completed a purchase, automatically mark them as being a payer.
        // TODO: Review which service is originally responsible for tracking this
        Embrace.getInstance().setUserAsPayer();

        this.purchaseMomentId = null;
        return true;
    }

    /**
     * Ends the purchase moment and generates an info log message that indicates that the purchase completed.
     * <p>
     * This method should be called once the purchase order has been confirmed.
     *
     * @return True if the operation was successful; false otherwise.
     */
    public boolean purchaseComplete() {
        return purchaseComplete(null);
    }

    /**
     * Ends the purchase moment and generates an error log message that indicates that the purchase failed.
     * <p>
     * This method should be called once the purchase order has been confirmed.
     *
     * @param msg A message that explains the reason for why this operation failed. This value is optional and, if
     *            provided, will associate the value as a property of the error log message.
     * @return True if the operation was successful; false otherwise.
     */
    public boolean purchaseFail(String msg) {
        return purchaseFail(msg, null);
    }


    /**
     * Ends the purchase moment and generates an error log message that indicates that the purchase failed.
     * <p>
     * This method should be called once the purchase order has been confirmed.
     *
     * @param msg        A message that explains the reason for why this operation failed. This value is optional and, if
     *                   provided, will associate the value as a property of the error log message.
     * @param properties Custom properties to pass in as part of the purchase failure
     * @return True if the operation was successful; false otherwise.
     */
    public boolean purchaseFail(String msg, Map<String, Object> properties) {
        if (this.purchaseMomentId == null) {
            InternalStaticEmbraceLogger.logError("Purchase wasn't started.");
            return false;
        } else if (msg != null) {
            this.purchaseProps.put(PROP_MESSAGE, msg);
        }

        String errorLogMsg;

        if (TextUtils.isEmpty(msg)) {
            errorLogMsg = "A failure occurred during purchase.";
        } else {
            errorLogMsg = "A failure occurred during purchase: " + msg;
        }

        sendMomentEndEvent(MOMENT_PURCHASE, this.purchaseMomentId, properties);
        sendLogError(errorLogMsg, false, this.purchaseProps);

        this.purchaseMomentId = null;
        return true;
    }


}
