package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import io.embrace.android.embracesdk.utils.optional.Optional;
import kotlin.Unit;
import kotlin.jvm.functions.Function1;

import com.google.gson.annotations.SerializedName;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents a particular user's session within the app.
 */
@InternalApi
public class Session {

    /**
     * Enum to discriminate the different ways a session can start / end
     */
    enum SessionLifeEventType {
        @SerializedName("s")
        STATE,
        @SerializedName("m")
        MANUAL,
        @SerializedName("t")
        TIMED
    }

    /**
     * A unique ID which identifies the session.
     */
    @SerializedName("id")
    private final String sessionId;
    /**
     * The time that the session started.
     */
    @SerializedName("st")
    private final Long startTime;
    /**
     * The time that the session ended.
     */
    @SerializedName("et")
    private final Long endTime;
    /**
     * The ordinal of the session, starting from 1.
     */
    @SerializedName("sn")
    private final Integer number;

    @SerializedName("ht")
    private final Long lastHeartbeatTime;

    @SerializedName("tt")
    private final Long terminationTime;

    @SerializedName("ls")
    private final String lastState;

    @SerializedName("ba")
    private final Double startingBatteryLevel;

    @SerializedName("ce")
    private final Boolean endedCleanly;

    @SerializedName("tr")
    private final Boolean receivedTermination;

    @SerializedName("cs")
    private final Boolean coldStart;

    @SerializedName("ss")
    private final List<String> eventIds;

    @SerializedName("t")
    private final String sessionType;

    @SerializedName("il")
    private final List<String> infoLogIds;

    @SerializedName("wl")
    private final List<String> warningLogIds;

    @SerializedName("el")
    private final List<String> errorLogIds;

    @SerializedName("lic")
    private final Integer infoLogsAttemptedToSend;

    @SerializedName("lwc")
    private final Integer warnLogsAttemptedToSend;

    @SerializedName("lec")
    private final Integer errorLogsAttemptedToSend;

    @SerializedName("e")
    private final ExceptionError exceptionError;

    @SerializedName("ri")
    private final String crashReportId;

    @SerializedName("em")
    private final SessionLifeEventType endType;

    @SerializedName("sm")
    private final SessionLifeEventType startType;

    @SerializedName("oc")
    private final List<Orientation> orientations;

    @SerializedName("sp")
    private final Map<String, String> properties;

    @SerializedName("sd")
    private final Long startupDuration;

    @SerializedName("sdt")
    private final Long startupThreshold;

    @SerializedName("si")
    private final Long sdkStartupDuration;

    @SerializedName("sst")
    private final List<StartupStacktrace> startupStacktraces;

    @SerializedName("ue")
    private final Integer unhandledExceptions;

    @SerializedName("nst")
    private final List<NativeThreadSample> nativeSampleTicks;

    /**
     * Beta feature data that was captured during this session
     */
    @SerializedName("bf")
    private final BetaFeatures betaFeatures;

    @SerializedName("sb")
    private final Map<String, String> symbols;

    private transient final UserInfo user;

    public String getSessionId() {
        return sessionId;
    }

    public UserInfo getUser() {
        return user;
    }

    public Long getStartTime() {
        return startTime;
    }

    public Long getEndTime() {
        return endTime;
    }

    public Integer getNumber() {
        return number;
    }

    public Long getLastHeartbeatTime() {
        return lastHeartbeatTime;
    }

    public Long getTerminationTime() {
        return terminationTime;
    }

    public String getLastState() {
        return lastState;
    }

    public Double getStartingBatteryLevel() {
        return startingBatteryLevel;
    }

    public Boolean isEndedCleanly() {
        return endedCleanly;
    }

    public Boolean isReceivedTermination() {
        return receivedTermination;
    }

    public Boolean isColdStart() {
        return coldStart;
    }

    public List<String> getEventIds() {
        return eventIds;
    }

    public String getSessionType() {
        return sessionType;
    }

    public List<String> getInfoLogIds() {
        return infoLogIds;
    }

    public List<String> getWarningLogIds() {
        return warningLogIds;
    }

    public List<String> getErrorLogIds() {
        return errorLogIds;
    }

    public Integer getInfoLogsAttemptedToSend() {
        return infoLogsAttemptedToSend;
    }

    public Integer getWarnLogsAttemptedToSend() {
        return warnLogsAttemptedToSend;
    }

    public Integer getErrorLogsAttemptedToSend() {
        return errorLogsAttemptedToSend;
    }

    public ExceptionError getExceptionError() {
        return exceptionError;
    }

    public String getCrashReportId() {
        return crashReportId;
    }

    public SessionLifeEventType getEndType() {
        return endType;
    }

    public SessionLifeEventType getStartType() {
        return startType;
    }

    public List<Orientation> getOrientations() {
        return orientations;
    }

    public Map<String, String> getProperties() {
        return properties;
    }

    public Long getStartupDuration() {
        return startupDuration;
    }

    public Long getStartupThreshold() {
        return startupThreshold;
    }

    public Long getSdkStartupDuration() {
        return sdkStartupDuration;
    }

    public List<StartupStacktrace> getStartupStacktraces() {
        return startupStacktraces;
    }

    public Integer getUnhandledExceptions() {
        return unhandledExceptions;
    }

    List<NativeThreadSample> getNativeSampleTicks() {
        return nativeSampleTicks;
    }

    Map<String, String> getSymbols() {
        return symbols;
    }

    BetaFeatures getBetaFeatures() {
        return betaFeatures;
    }

    Session(Builder builder) {
        sessionId = builder.sessionId;
        user = builder.user;
        startTime = builder.startTime;
        endTime = builder.endTime;
        number = builder.number;
        lastHeartbeatTime = builder.lastHeartbeatTime;
        terminationTime = builder.terminationTime;
        lastState = builder.lastState;
        startingBatteryLevel = builder.startingBatteryLevel;
        endedCleanly = builder.endedCleanly;
        receivedTermination = builder.receivedTermination;
        coldStart = builder.coldStart;
        eventIds = builder.eventIds;
        sessionType = builder.sessionType;
        infoLogIds = builder.infoLogIds;
        warningLogIds = builder.warningLogIds;
        errorLogIds = builder.errorLogIds;
        exceptionError = builder.exceptionError;
        infoLogsAttemptedToSend = builder.infoLogsAttemptedToSend;
        warnLogsAttemptedToSend = builder.warnLogsAttemptedToSend;
        errorLogsAttemptedToSend = builder.errorLogsAttemptedToSend;
        crashReportId = builder.crashReportId;
        startType = builder.startType;
        endType = builder.endType;
        orientations = builder.orientations;
        properties = builder.properties;
        startupDuration = builder.startupDuration;
        startupThreshold = builder.startupThreshold;
        sdkStartupDuration = builder.sdkStartupDuration;
        startupStacktraces = builder.startupStacktraces;
        unhandledExceptions = builder.unhandledExceptions;
        nativeSampleTicks = builder.nativeSampleTicks;
        symbols = builder.symbols;
        betaFeatures = builder.betaFeatures;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(Session copy) {
        Builder builder = new Builder();
        builder.sessionId = copy.getSessionId();
        builder.user = copy.getUser();
        builder.startTime = copy.getStartTime();
        builder.endTime = copy.getEndTime();
        builder.number = copy.getNumber();
        builder.lastHeartbeatTime = copy.getLastHeartbeatTime();
        builder.terminationTime = copy.getTerminationTime();
        builder.lastState = copy.getLastState();
        builder.startingBatteryLevel = copy.getStartingBatteryLevel();
        builder.endedCleanly = copy.isEndedCleanly();
        builder.receivedTermination = copy.isReceivedTermination();
        builder.coldStart = copy.isColdStart();
        builder.eventIds = copy.getEventIds();
        builder.sessionType = copy.getSessionType();
        builder.infoLogIds = copy.getInfoLogIds();
        builder.warningLogIds = copy.getWarningLogIds();
        builder.errorLogIds = copy.getErrorLogIds();
        builder.infoLogsAttemptedToSend = copy.getInfoLogsAttemptedToSend();
        builder.warnLogsAttemptedToSend = copy.getWarnLogsAttemptedToSend();
        builder.errorLogsAttemptedToSend = copy.getErrorLogsAttemptedToSend();
        builder.exceptionError = copy.getExceptionError();
        builder.crashReportId = copy.getCrashReportId();
        builder.startType = copy.getStartType();
        builder.endType = copy.getEndType();
        builder.orientations = copy.getOrientations();
        builder.properties = copy.getProperties();
        builder.startupDuration = copy.getStartupDuration();
        builder.startupThreshold = copy.getStartupThreshold();
        builder.sdkStartupDuration = copy.getSdkStartupDuration();
        builder.startupStacktraces = copy.getStartupStacktraces();
        builder.unhandledExceptions = copy.getUnhandledExceptions();
        builder.nativeSampleTicks = copy.getNativeSampleTicks();
        builder.symbols = copy.getSymbols();
        builder.betaFeatures = copy.getBetaFeatures();
        return builder;
    }

    @InternalApi
    public static final class Builder {

        String sessionId;
        UserInfo user;
        Long startTime;
        Long endTime;
        Integer number;
        Long lastHeartbeatTime;
        Long terminationTime;
        String lastState;
        Double startingBatteryLevel;
        Boolean endedCleanly;
        Boolean receivedTermination;
        Boolean coldStart;
        List<String> eventIds;
        String sessionType;
        List<String> infoLogIds;
        List<String> warningLogIds;
        List<String> errorLogIds;
        Integer infoLogsAttemptedToSend;
        Integer warnLogsAttemptedToSend;
        Integer errorLogsAttemptedToSend;
        ExceptionError exceptionError;
        String crashReportId;
        SessionLifeEventType endType;
        SessionLifeEventType startType;
        List<Orientation> orientations;
        Map<String, String> properties;
        Long startupDuration;
        Long startupThreshold;
        Long sdkStartupDuration;
        List<StartupStacktrace> startupStacktraces;
        Integer unhandledExceptions;
        List<NativeThreadSample> nativeSampleTicks;
        Map<String, String> symbols;
        BetaFeatures betaFeatures;

        Builder() {
        }

        @NonNull
        public Builder withSessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        @NonNull
        public Builder withUserInfo(Optional<UserInfo> userInfo) {
            if (userInfo.isPresent()) {
                this.user = userInfo.get();
            }
            return this;
        }

        @NonNull
        public Builder withStartTime(Long startTime) {
            this.startTime = startTime;
            return this;
        }

        @NonNull
        public Builder withEndTime(Long endTime) {
            this.endTime = endTime;
            return this;
        }

        @NonNull
        public Builder withNumber(Integer number) {
            this.number = number;
            return this;
        }

        @NonNull
        public Builder withLastHeartbeatTime(Long lastHeartbeatTime) {
            this.lastHeartbeatTime = lastHeartbeatTime;
            return this;
        }

        @NonNull
        public Builder withTerminationTime(@Nullable Long terminationTime) {
            this.terminationTime = terminationTime;
            return this;
        }

        @NonNull
        public Builder withLastState(String lastState) {
            this.lastState = lastState;
            return this;
        }

        @NonNull
        public Builder withStartingBatteryLevel(Optional<Float> startingBatteryLevel) {
            if (startingBatteryLevel.isPresent()) {
                this.startingBatteryLevel = Double.valueOf(startingBatteryLevel.get());
            }
            return this;
        }

        @NonNull
        public Builder withEndedCleanly(boolean endedCleanly) {
            this.endedCleanly = endedCleanly;
            return this;
        }

        @NonNull
        public Builder withReceivedTermination(@Nullable Boolean receivedTermination) {
            this.receivedTermination = receivedTermination;
            return this;
        }

        @NonNull
        public Builder withColdStart(boolean coldStart) {
            this.coldStart = coldStart;
            return this;
        }

        @NonNull
        public Builder withEventIds(@Nullable List<String> eventIds) {
            this.eventIds = (eventIds != null) ? new ArrayList<>(eventIds) : null;
            return this;
        }

        @NonNull
        public Builder withSessionType(String sessionType) {
            this.sessionType = sessionType;
            return this;
        }

        @NonNull
        public Builder withInfoLogIds(@Nullable List<String> infoLogIds) {
            this.infoLogIds = (infoLogIds != null) ? new ArrayList<>(infoLogIds) : null;
            return this;
        }

        @NonNull
        public Builder withWarningLogIds(@Nullable List<String> warningLogIds) {
            this.warningLogIds = (warningLogIds != null) ? new ArrayList<>(warningLogIds) : null;
            return this;
        }

        @NonNull
        public Builder withErrorLogIds(@Nullable List<String> errorLogIds) {
            this.errorLogIds = (errorLogIds != null) ? new ArrayList<>(errorLogIds) : null;
            return this;
        }

        @NonNull
        public Builder withInfoLogsAttemptedToSend(Integer infoLogsAttemptedToSend) {
            this.infoLogsAttemptedToSend = infoLogsAttemptedToSend;
            return this;
        }

        @NonNull
        public Builder withWarnLogsAttemptedToSend(Integer warnLogsAttemptedToSend) {
            this.warnLogsAttemptedToSend = warnLogsAttemptedToSend;
            return this;
        }

        @NonNull
        public Builder withErrorLogsAttemptedToSend(Integer errorLogsAttemptedToSend) {
            this.errorLogsAttemptedToSend = errorLogsAttemptedToSend;
            return this;
        }

        @NonNull
        public Builder withExceptionErrors(ExceptionError exceptionError) {
            this.exceptionError = exceptionError;
            return this;
        }

        @NonNull
        public Builder withCrashReportId(String crashReportId) {
            this.crashReportId = crashReportId;
            return this;
        }

        @NonNull
        public Builder withEndType(SessionLifeEventType endType) {
            this.endType = endType;
            return this;
        }

        @NonNull
        public Builder withStartType(SessionLifeEventType startType) {
            this.startType = startType;
            return this;
        }

        @NonNull
        public Builder withProperties(Map<String, String> properties) {
            this.properties = properties;
            return this;
        }

        @NonNull
        public Builder withOrientations(@Nullable List<Orientation> orientations) {
            this.orientations = (orientations != null) ? new ArrayList<>(orientations) : null;
            return this;
        }

        @NonNull
        public Builder withStartupDuration(Long startupDuration) {
            this.startupDuration = startupDuration;
            return this;
        }

        @NonNull
        public Builder withStartupThreshold(Long startupThreshold) {
            this.startupThreshold = startupThreshold;
            return this;
        }

        @NonNull
        public Builder withSdkStartupDuration(Long sdkStartupDuration) {
            this.sdkStartupDuration = sdkStartupDuration;
            return this;
        }

        @NonNull
        public Builder withStartupStacktraces(@Nullable List<StartupStacktrace> startupStacktraces) {
            this.startupStacktraces = (startupStacktraces != null) ? new ArrayList<>(startupStacktraces) : null;
            return this;
        }

        @NonNull
        public Builder withUnhandledExceptions(int unhandledExceptions) {
            this.unhandledExceptions = unhandledExceptions;
            return this;
        }

        @NonNull
        public Builder withNativeSampleTicks(@NonNull List<NativeThreadSample> samples) {
            this.nativeSampleTicks = new ArrayList<>(samples);
            return this;
        }

        @NonNull
        public Builder withNativeSymbols(@Nullable Map<String, String> symbols) {
            this.symbols = (symbols != null) ? new HashMap<>(symbols) : null;
            return this;
        }

        @NonNull
        public Builder withBetaFeatures(@NonNull Function1<BetaFeatures, Unit> function) {
            if (betaFeatures == null) {
                betaFeatures = new BetaFeatures();
            }
            function.invoke(betaFeatures);
            return this;
        }

        @NonNull
        public Builder disableBetaFeatures() {
            betaFeatures = null;
            return this;
        }

        @Nullable
        public Map<String, String> getProperties() {
            return properties;
        }

        @NonNull
        public Session build() {
            return new Session(this);
        }
    }
}
