package io.embrace.android.embracesdk

import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import io.embrace.android.embracesdk.utils.optional.Optional

/**
 * The name of the file containing the last cached session.
 */
private const val SESSION_FILE_NAME = "last_session.json"

/**
 * The name of the file containing the previous cached session.
 */
private const val PREVIOUS_SESSION_FILE_NAME = "previous_session.json"

/**
 * This class encapsulates all functionality of session caching.
 *
 * The idea is that caching sessions should mandatory go through this class.
 * Also, this class contains state on lock/unlock. This is needed in order to
 * maintain certain order when native crashes are involved.
 */
internal class SessionCacheManager(
    localConfig: LocalConfig,
    private val cacheService: CacheService,
    private val apiClient: ApiClient,
    private val logger: InternalEmbraceLogger
) {

    /**
     * It determines if previous session cache is locked.
     */
    @Volatile
    var previousSessionCacheLocked = false
        private set

    init {
        // if NDK crash capture is enabled we will lock previous session cache by default
        // this is because in case anyone else tries to update previous session cache, we want to wait
        // until native crash is found so not to lose previous cached session
        previousSessionCacheLocked = localConfig.isNdkEnabled
    }

    /**
     * Caches a generated session message, with performance information generated up to the current
     * point.
     */
    fun cacheCurrentSessionMessage(sessionMessage: SessionMessage): Boolean {
        return try {
            cacheService.cacheObject(
                SESSION_FILE_NAME, sessionMessage,
                SessionMessage::class.java
            )
            logger.logDebug("Current active session successfully cached.")

            true
        } catch (ex: Exception) {
            logger.logWarning("Failed to cache current active session", ex)
            false
        }
    }

    /**
     * It sends to our servers previous session that was stored in this cache.
     *
     * If cache is locked it will not perform anything.
     */
    fun sendPreviousCachedSession(): Boolean {
        if (!previousSessionCacheLocked) {
            logger.logDebug("Cache is unlocked for previous session. Will try to send previous cached session.")
            return try {
                val previousSession: Optional<SessionMessage> = fetchPreviousSessionMessage()
                if (previousSession.isPresent) {
                    apiClient.sendSession(previousSession.get())
                    logger.logDebug("Previous cached session successfully sent.")
                    removePreviousSessionMessage()

                    true
                } else {
                    logger.logDebug("Previous cached session not sent because it did not exist.")
                    false
                }
            } catch (ex: Exception) {
                logger.logWarning("Failed to send previous cached session message.", ex)
                false
            }
        } else {
            logger.logDebug("Cache is locked for previous session. Will not send previous cached session.")
            return false
        }
    }

    /**
     * It will replace previous cached session by given session.
     */
    fun updatePreviousSessionMessageCache(sessionMessage: SessionMessage): Boolean {
        return try {
            cacheService.cacheObject(
                PREVIOUS_SESSION_FILE_NAME,
                sessionMessage,
                SessionMessage::class.java
            )
            logger.logDebug("Previous session successfully cached.")

            true
        } catch (ex: java.lang.Exception) {
            logger.logWarning("Failed to cache previous session", ex)
            false
        }
    }

    /**
     * It removes current session message from cache.
     */
    fun removeCurrentSessionMessage(): Boolean {
        return try {
            cacheService.deleteObject(SESSION_FILE_NAME)
            logger.logDebug("Successfully removed current active session from cache.")
            true
        } catch (ex: java.lang.Exception) {
            logger.logDebug("Failed to remove current session message from cache.", ex)
            false
        }
    }

    /**
     * It fetches previous session from cache.
     */
    fun fetchPreviousSessionMessage(): Optional<SessionMessage> {
        try {
            val previousSessionMessage = cacheService.loadObject(PREVIOUS_SESSION_FILE_NAME, SessionMessage::class.java)
            logger.logDebug("Successfully fetched previous session message.")
            return previousSessionMessage
        } catch (ex: Exception) {
            logger.logWarning("Failed to load previous cached session message", ex)
        }

        return Optional.absent()
    }

    fun unlockPreviousSessionCache() {
        // unlock
        previousSessionCacheLocked = false
        logger.logDebug("Previous session cache successfully unlocked.")
    }

    /**
     * It stashes cached session. Meaning it will copy current session to previous session file name.
     */
    fun stashPreviousSession(): Boolean {
        return try {
            val stashed = cacheService.moveObject(SESSION_FILE_NAME, PREVIOUS_SESSION_FILE_NAME)
            if (stashed) {
                logger.logDebug("Successfully stashed previous session.")
            } else {
                logger.logDebug("Failed to stashe previous session.")
            }

            stashed
        } catch (ex: java.lang.Exception) {
            logger.logWarning("Failed to stash previous session message.", ex)
            false
        }
    }

    /**
     * It removes previous session message from cache.
     */
    private fun removePreviousSessionMessage(): Boolean {
        return try {
            val deleted = cacheService.deleteObject(PREVIOUS_SESSION_FILE_NAME)
            if (deleted) {
                logger.logDebug("Successfully removed previous session message from cache.")
            } else {
                logger.logDebug("Failed to remove previous session message from cache.")
            }

            deleted
        } catch (ex: java.lang.Exception) {
            logger.logDebug("Failed to remove previous session message from cache.", ex)
            false
        }
    }
}
