package io.embrace.android.embracesdk

import android.app.Activity
import io.embrace.android.embracesdk.EmbraceSessionService.SESSION_CACHING_INTERVAL
import io.embrace.android.embracesdk.Session.SessionLifeEventType
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger.Companion.logDeveloper
import io.embrace.android.embracesdk.utils.exceptions.Unchecked
import io.embrace.android.embracesdk.utils.optional.Optional
import java.io.Closeable
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.TimeUnit

/**
 * Signals to the API the start of a session.
 */
private const val SESSION_START_TYPE = "st"

/**
 * Signals to the API the end of a session.
 */
private const val SESSION_END_TYPE = "en"

/**
 * ApiClient timeout for the active session send service
 */
private const val SEND_SESSION_API_CLIENT_TIMEOUT = 2

internal class SessionHandler(
    private val logger: InternalEmbraceLogger,
    private val configService: ConfigService,
    private val preferencesService: PreferencesService,
    private val powerService: PowerService,
    private val userService: UserService,
    private val networkConnectivityService: NetworkConnectivityService,
    private val metadataService: MetadataService,
    private val apiClient: ApiClient,
    private val gatingService: GatingService,
    private val localConfig: LocalConfig,
    private val breadcrumbService: BreadcrumbService,
    private val activityService: ActivityService,
    private val ndkService: NdkService,
    private val eventService: EventService,
    private val remoteLogger: EmbraceRemoteLogger,
    private val exceptionService: EmbraceExceptionService,
    private val startupTracingService: StartupTracingService?,
    private val performanceInfoService: PerformanceInfoService,
    private val memoryCleanerService: MemoryCleanerService,
    private val sessionCacheManager: SessionCacheManager,
    private val clock: Clock
) : Closeable {

    private var automaticSessionStopper = ScheduledWorker.ofSingleThread("Session Closer Service")
    private var sessionPeriodicCacheWorker =
        ScheduledWorker.ofSingleThread("Session Caching Service")
    private val sessionBackgroundWorker: BackgroundWorker by lazy {
        BackgroundWorker.ofSingleThread("Session")
    }
    private val sessionStartListeners = CopyOnWriteArrayList<SessionStartListener>()
    private val sessionEndListeners = CopyOnWriteArrayList<SessionEndListener>()

    /**
     * It performs all corresponding operations in order to start a session.
     */
    fun onSessionStarted(
        coldStart: Boolean,
        startType: SessionLifeEventType,
        sessionProperties: EmbraceSessionProperties,
        automaticSessionCloserCallback: Runnable,
        cacheCallback: Runnable
    ): SessionMessage? {

        if (!isAllowedToStart()) {
            logger.logDebug("Session not allowed to start.")
            return null
        }

        logDeveloper("SessionHandler", "Session Started")
        val session = buildStartSession(Uuid.getEmbUuid(), coldStart, startType, sessionProperties)
        logDeveloper("SessionHandler", "SessionId = ${session.sessionId}")

        // Record the connection type at the start of the session.
        networkConnectivityService.networkStatusOnSessionStarted(session.startTime)

        val sessionMessage = buildStartSessionMessage(session)

        metadataService.setActiveSessionId(session.sessionId)

        // sanitize start session message before send it to backend
        val sanitizedSession = gatingService.gateSessionMessage(sessionMessage)
        logger.logDebug("Start session successfully sanitized.")

        apiClient.sendSession(sanitizedSession)
        logger.logDebug("Start session successfully sent.")

        handleAutomaticSessionStopper(automaticSessionCloserCallback)
        addViewBreadcrumbForResumedSession()
        startPeriodicCaching(cacheCallback)
        if (localConfig.isNdkEnabled) {
            ndkService.updateSessionId(session.sessionId)
        }

        return sessionMessage
    }

    /**
     * It performs all corresponding operations in order to end a session.
     */
    fun onSessionEnded(
        endType: SessionLifeEventType,
        originSession: Session?,
        sessionProperties: EmbraceSessionProperties,
        sdkStartupDuration: Long
    ) {
        logger.logDebug("Will try to run end session full.")
        if (localConfig.configurations.getSessionConfig()
            .getAsyncEnd() || configService.config.endSessionInBackgroundThread()
        ) {
            sessionBackgroundWorker.submit {
                runEndSessionFull(endType, originSession, sessionProperties, sdkStartupDuration)
            }
        } else {
            runEndSessionFull(endType, originSession, sessionProperties, sdkStartupDuration)
        }
    }

    /**
     * Called when a regular crash happens. It will build a session message with associated crashId,
     * and send it to our servers.
     */
    fun onCrash(
        originSession: Session,
        crashId: String,
        sessionProperties: EmbraceSessionProperties,
        sdkStartupDuration: Long
    ) {
        logger.logDebug("Will try to run end session for crash.")
        runEndSessionForCrash(
            originSession,
            crashId,
            sessionProperties,
            sdkStartupDuration,
            SessionLifeEventType.STATE
        )
    }

    /**
     * Called when periodic cache update needs to be performed.
     * It will update current session 's cache state.
     *
     * Note that the session message will not be sent to our servers.
     */
    fun onPeriodicCacheActiveSession(
        activeSession: Session?,
        sessionProperties: EmbraceSessionProperties,
        sdkStartupDuration: Long
    ) {
        activeSession?.let {
            logger.logDebug("Will try to run end session for caching.")
            runEndSessionForCaching(
                activeSession,
                sessionProperties,
                sdkStartupDuration,
                SessionLifeEventType.STATE
            )
        } ?: kotlin.run {
            logger.logDebug("Will no perform active session caching because there is no active session available.")
        }
    }

    fun addSessionStartListeners(listener: Collection<SessionStartListener>) {
        sessionStartListeners.addAll(listener)
    }

    fun addSessionEndListeners(listener: Collection<SessionEndListener>) {
        sessionEndListeners.addAll(listener)
    }

    override fun close() {
        stopPeriodicSessionCaching()
        stopAutomaticSessionStopper()
        stopSessionEndingWorker()
    }

    private fun stopAutomaticSessionStopper() {
        automaticSessionStopper?.let {
            logger.logDebug("Stopping automatic session closer.")
            it.close()
        }
    }

    private fun stopPeriodicSessionCaching() {
        sessionPeriodicCacheWorker?.let {
            logger.logDebug("Stopping session caching.")
            it.close()
        }
    }

    private fun stopSessionEndingWorker() {
        sessionBackgroundWorker.close()
    }

    /**
     * If maximum timeout session is set through config, then this method starts automatic session
     * stopper worker, so session timeouts at given time.
     */
    private fun handleAutomaticSessionStopper(automaticSessionCloserCallback: Runnable) {
        // If getMaxSessionSeconds is not null, schedule the session stopper.
        with(localConfig.configurations.getSessionConfig()) {
            val shouldStartAutomaticSessionStopper = getMaxSessionSecondsAllowed().isPresent
            if (shouldStartAutomaticSessionStopper) {
                logger.logDebug("Will start automatic session stopper.")
                startAutomaticSessionStopper(
                    automaticSessionCloserCallback,
                    getMaxSessionSecondsAllowed().get()
                )
            } else {
                logger.logDebug("Maximum session timeout not set on config. Will not start automatic session stopper.")
            }
        }
    }

    /**
     * It determines if we are allowed to build an end session message.
     */
    private fun isAllowedToEnd(endType: SessionLifeEventType, activeSession: Session?): Boolean {
        if (activeSession == null) {
            logger.logDebug("No active session found. Session is not allowed to end.")
            return false
        }

        return when (endType) {
            SessionLifeEventType.STATE -> {
                // state sessions are always allowed to be ended
                logger.logDebug("Session is STATE, it is always allowed to end.")
                true
            }
            SessionLifeEventType.MANUAL, SessionLifeEventType.TIMED -> {
                logger.logDebug("Session is either MANUAL or TIMED.")
                if (!configService.isSessionControlEnabled) {
                    logger.logWarning("Session control disabled from remote configuration. Session is not allowed to end.")
                    false
                } else if (endType == SessionLifeEventType.MANUAL && ((clock.now() - activeSession.startTime) < EmbraceSessionService.minSessionTime)) {
                    // If less than 5 seconds, then the session cannot be finished manually.
                    logger.logError("The session has to be of at least 5 seconds to be ended manually.")
                    false
                } else {
                    logger.logDebug("Session allowed to end.")
                    true
                }
            }
        }
    }

    private fun buildStartSession(
        id: String,
        coldStart: Boolean,
        startType: SessionLifeEventType,
        sessionProperties: EmbraceSessionProperties
    ): Session {
        val builder = Session.newBuilder()
            .withSessionId(id)
            .withStartTime(clock.now())
            .withNumber(incrementAndGetSessionNumber())
            .withColdStart(coldStart)
            .withStartType(startType)
            .withProperties(sessionProperties.get())
            .withSessionType(SESSION_START_TYPE)
            .withStartingBatteryLevel(powerService.latestBatteryLevel)
            .withUserInfo(userService.loadUserInfoFromDisk())

        // invoke callbacks - all data should be collected in here ultimately to reduce
        // the responsibilities of this service.
        for (listener in sessionStartListeners) {
            listener.onSessionStart(builder)
        }

        return builder.build()
    }

    private fun buildEndSessionMessage(
        originSession: Session,
        endedCleanly: Boolean,
        forceQuit: Boolean,
        crashId: String?,
        endType: SessionLifeEventType,
        sessionProperties: EmbraceSessionProperties,
        sdkStartupDuration: Long
    ): SessionMessage {

        val startTime: Long = originSession.startTime
        val endTime = clock.now()

        // TODO future: we should extract more of this logic out to the listeners.
        // https://app.clickup.com/t/26ba651
        val builder = Session.newBuilder(originSession)
            .withEndedCleanly(endedCleanly)
            .withLastState(getApplicationState())
            .withSessionType(SESSION_END_TYPE)
            .withEventIds(eventService.findEventIdsForSession(startTime, endTime))
            .withInfoLogIds(remoteLogger.findInfoLogIds(startTime, endTime))
            .withWarningLogIds(remoteLogger.findWarningLogIds(startTime, endTime))
            .withErrorLogIds(remoteLogger.findErrorLogIds(startTime, endTime))
            .withInfoLogsAttemptedToSend(remoteLogger.infoLogsAttemptedToSend)
            .withWarnLogsAttemptedToSend(remoteLogger.warnLogsAttemptedToSend)
            .withErrorLogsAttemptedToSend(remoteLogger.errorLogsAttemptedToSend)
            .withExceptionErrors(exceptionService.currentExceptionError)
            .withLastHeartbeatTime(clock.now())
            .withProperties(sessionProperties.get())
            .withEndType(endType)
            .withUnhandledExceptions(remoteLogger.unhandledExceptionsSent)

        // invoke callbacks - all data should be collected in here ultimately to reduce
        // the responsibilities of this service.
        for (listener in sessionEndListeners) {
            listener.onSessionEnd(builder)
        }

        if (!crashId.isNullOrEmpty()) {
            // if it's a crash session, then add the stacktrace to the session payload
            builder.withCrashReportId(crashId)
        }
        if (forceQuit) {
            builder
                .withTerminationTime(endTime)
                .withReceivedTermination(true)
        } else {
            // We don't set end time for force-quit, as the API interprets this to be a clean
            // termination
            builder.withEndTime(endTime)
        }

        val startupEventInfo = eventService.startupMomentInfo
        if (originSession.isColdStart) {
            builder.withSdkStartupDuration(sdkStartupDuration)
            if (startupEventInfo != null) {
                builder.withStartupDuration(startupEventInfo.duration)
                builder.withStartupThreshold(startupEventInfo.threshold)
            }
            startupTracingService?.let {
                builder.withStartupStacktraces(it.stacktraces)
            }
        }

        val endSession = builder.build()
        return SessionMessage.newBuilder()
            .withUserInfo(endSession.user)
            .withAppInfo(metadataService.appInfo)
            .withDeviceInfo(metadataService.deviceInfo)
            .withPerformanceInfo(
                performanceInfoService.getSessionPerformanceInfo(
                    startTime,
                    endTime
                )
            )
            .withBreadcrumbs(breadcrumbService.getBreadcrumbs(startTime, endTime))
            .withSession(endSession)
            .build()
    }

    private fun buildStartSessionMessage(session: Session) = SessionMessage.newBuilder()
        .withSession(session)
        .withDeviceInfo(metadataService.deviceInfo)
        .withAppInfo(metadataService.appInfo)
        .build()

    /**
     * It builds an end active session message, it sanitizes it, it performs all types of memory cleaning,
     * it updates cache and it sends it to our servers.
     * It also stops periodic caching and automatic session stopper.
     */
    private fun runEndSessionFull(
        endType: SessionLifeEventType,
        originSession: Session?,
        sessionProperties: EmbraceSessionProperties,
        sdkStartupDuration: Long
    ) {
        if (!isAllowedToEnd(endType, originSession)) {
            logger.logDebug("Session not allowed to end.")
            return
        }

        stopPeriodicSessionCaching()
        stopAutomaticSessionStopper()

        if (configService.isMessageTypeDisabled(MessageType.SESSION)) {
            logger.logWarning("Session messages disabled. Ignoring all Sessions.")
            return
        }

        val fullEndSessionMessage = buildEndSessionMessage(
            /* we are previously checking in allowSessionToEnd that originSession != null */
            originSession!!,
            endedCleanly = true,
            forceQuit = false,
            null,
            endType,
            sessionProperties,
            sdkStartupDuration
        )
        logger.logDeveloper("SessionHandler", "End session message=$fullEndSessionMessage")

        // Cache the active session to ensure that the cache agrees with the data we are
        // about to send. We do this since:
        //
        // 1. If the session data is sent, but the completion of the send is not registered
        //    by the SDK, then we will re-send an incomplete set of session data later.
        // 2. If the app crashes before we send the session message, the data since the last
        //    cache interval is lost.
        sessionCacheManager.cacheCurrentSessionMessage(fullEndSessionMessage)
        logger.logDebug("Full session message cached successfully.")

        // Clean every collection of those services which have collections in memory.
        memoryCleanerService.cleanServicesCollections(metadataService, exceptionService)
        logger.logDebug("Services collections successfully cleaned.")
        try {
            // During power saving mode, requests are hanged forever causing an ANR when
            // Foregrounded, so the Timeout prevents this scenario from happening.

            // Sanitize session message
            val sanitizedSessionMessage = gatingService.gateSessionMessage(fullEndSessionMessage)
            logger.logDeveloper(
                "SessionHandler",
                "Sanitized End session message=$sanitizedSessionMessage"
            )

            apiClient.sendSession(sanitizedSessionMessage)[SEND_SESSION_API_CLIENT_TIMEOUT.toLong(), TimeUnit.SECONDS]
            logger.logDebug("Session message sent successfully.")

            sessionCacheManager.removeCurrentSessionMessage()
            logger.logDebug("Current session has been successfully removed from cache.")
        } catch (ex: Exception) {
            logger.logInfo(
                "Failed to send session end message. Embrace will store the " +
                    "session message and attempt to deliver it at a future date."
            )
        }

        sessionProperties.clearTemporary()
        logger.logDebug("Session properties successfully temporary cleared.")
    }

    /**
     * It builds an end active session message, it sanitizes it, it updates cache and it sends it to our servers synchronously.
     *
     * This is because when a crash happens, we do not have the ability to start a background
     * thread because the JVM will soon kill the process. So we force the request to be performed
     * in main thread.
     *
     * Note that this may cause ANRs. In the future we should come up with a better approach.
     *
     * Also note that we do not perform any memory cleaning because since the app is about to crash,
     * we do not to waste time on those things.
     */
    private fun runEndSessionForCrash(
        originSession: Session,
        crashId: String,
        sessionProperties: EmbraceSessionProperties,
        sdkStartupDuration: Long,
        endType: SessionLifeEventType
    ) {
        if (!isAllowedToEnd(endType, originSession)) {
            logger.logDebug("Session not allowed to end.")
            return
        }

        val fullEndSessionMessage = buildEndSessionMessage(
            originSession,
            endedCleanly = false,
            forceQuit = false,
            crashId,
            endType,
            sessionProperties,
            sdkStartupDuration
        )
        logger.logDeveloper("SessionHandler", "End session message=$fullEndSessionMessage")

        // Sanitize session message
        val sanitizedSessionMessage = gatingService.gateSessionMessage(fullEndSessionMessage)
        logger.logDeveloper(
            "SessionHandler",
            "Sanitized End session message=$sanitizedSessionMessage"
        )

        Unchecked.wrap<String> {
            // perform session request synchronously
            apiClient.sendSession(
                sanitizedSessionMessage
            ).get()
        }
        sessionCacheManager.removeCurrentSessionMessage()
    }

    /**
     * It builds an end active session message and it updates cache.
     *
     * Note that it does not send the session to our servers.
     */
    private fun runEndSessionForCaching(
        activeSession: Session,
        sessionProperties: EmbraceSessionProperties,
        sdkStartupDuration: Long,
        endType: SessionLifeEventType
    ) {
        if (!isAllowedToEnd(endType, activeSession)) {
            logger.logDebug("Session not allowed to end.")
            return
        }

        val fullEndSessionMessage = buildEndSessionMessage(
            activeSession,
            endedCleanly = false,
            forceQuit = true,
            null,
            endType,
            sessionProperties,
            sdkStartupDuration
        )
        logger.logDeveloper("SessionHandler", "End session message=$fullEndSessionMessage")

        sessionCacheManager.cacheCurrentSessionMessage(fullEndSessionMessage)
    }

    /**
     * @return session number incremented by 1
     */
    private fun incrementAndGetSessionNumber(): Int {
        var sessionNumber: Int?
        if (preferencesService.sessionNumber.isPresent) {
            sessionNumber = preferencesService.sessionNumber.get() + 1
            preferencesService.setSessionNumber(sessionNumber)
        } else {
            sessionNumber = 1
            preferencesService.setSessionNumber(sessionNumber)
        }
        return sessionNumber
    }

    /**
     * It starts a background worker that will schedule a callback to automatically end the session.
     */
    private fun startAutomaticSessionStopper(
        automaticSessionStopperCallback: Runnable,
        maxSessionSeconds: Int
    ) {
        if (localConfig.configurations.getSessionConfig()
            .getAsyncEnd() || configService.config.endSessionInBackgroundThread()
        ) {
            logger.logWarning("Can't close the session. Automatic session closing disabled since async session send is enabled.")
            return
        }

        automaticSessionStopper = ScheduledWorker.ofSingleThread("Session Closer Service")
        this.automaticSessionStopper.scheduleAtFixedRate(
            automaticSessionStopperCallback,
            maxSessionSeconds.toLong(), maxSessionSeconds.toLong(), TimeUnit.SECONDS
        )
        logger.logDebug("Automatic session stopper successfully scheduled.")
    }

    /**
     * As when resuming the app a new session is created, the screen would be the same as the
     * one the app had when was backgrounded. In this case, the SDK skips the "duplicate view
     * breadcrumb" scenario. This method, forces the repetition of the view breadcrumb when the
     * app's being resumed.
     */
    private fun addViewBreadcrumbForResumedSession() {
        val screen: Optional<String> = breadcrumbService.lastViewBreadcrumbScreenName
        if (screen.isPresent) {
            breadcrumbService.forceLogView(screen.orNull(), clock.now())
        } else {
            val foregroundActivity: Optional<Activity> = activityService.foregroundActivity
            if (foregroundActivity.isPresent) {
                breadcrumbService.forceLogView(
                    foregroundActivity.get().localClassName,
                    clock.now()
                )
            }
        }
    }

    private fun isAllowedToStart(): Boolean {
        return if (configService.isMessageTypeDisabled(MessageType.SESSION)) {
            logger.logWarning("Session messages disabled. Ignoring all sessions.")
            false
        } else {
            logger.logDebug("Session is allowed to start.")
            true
        }
    }

    /**
     * It starts a background worker that will schedule a callback to do periodic caching.
     */
    private fun startPeriodicCaching(cacheCallback: Runnable) {
        sessionPeriodicCacheWorker = ScheduledWorker.ofSingleThread("Session Caching Service")
        this.sessionPeriodicCacheWorker.scheduleAtFixedRate(
            cacheCallback,
            0,
            SESSION_CACHING_INTERVAL.toLong(),
            TimeUnit.SECONDS
        )
        logger.logDebug("Periodic session cache successfully scheduled.")
    }

    private fun getApplicationState(): String {
        return if (activityService.isInBackground) EmbraceSessionService.APPLICATION_STATE_BACKGROUND else EmbraceSessionService.APPLICATION_STATE_ACTIVE
    }
}
