package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.annotations.SerializedName;

/**
 * The session message, containing the session itself, as well as performance information about the
 * device which occurred during the session.
 */
@InternalApi
public final class SessionMessage {

    /** The session information. */
    @SerializedName("s")
    private final Session session;

    /** The user information. */
    @SerializedName("u")
    private final UserInfo userInfo;

    /** The app information. */
    @SerializedName("a")
    private final AppInfo appInfo;

    /** The device information. */
    @SerializedName("d")
    private final DeviceInfo deviceInfo;

    /** The device's performance info, such as power, cpu, network. */
    @SerializedName("p")
    private final SessionPerformanceInfo performanceInfo;

    /** Breadcrumbs which occurred as part of this session. */
    @SerializedName("br")
    private final Breadcrumbs breadcrumbs;

    @SerializedName("v")
    private final int version = ApiClient.MESSAGE_VERSION;

    SessionMessage(Builder builder) {
        session = builder.session;
        userInfo = builder.userInfo;
        appInfo = builder.appInfo;
        deviceInfo = builder.deviceInfo;
        performanceInfo = builder.performanceInfo;
        breadcrumbs = builder.breadcrumbs;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(SessionMessage copy) {
        Builder builder = new Builder();
        builder.session = copy.getSession();
        builder.userInfo = copy.getUserInfo();
        builder.appInfo = copy.getAppInfo();
        builder.deviceInfo = copy.getDeviceInfo();
        builder.performanceInfo = copy.getPerformanceInfo();
        builder.breadcrumbs = copy.getBreadcrumbs();
        return builder;
    }

    public Session getSession() {
        return session;
    }

    public UserInfo getUserInfo() {
        return userInfo;
    }

    public AppInfo getAppInfo() {
        return appInfo;
    }

    public DeviceInfo getDeviceInfo() {
        return deviceInfo;
    }

    public SessionPerformanceInfo getPerformanceInfo() {
        return performanceInfo;
    }

    public Breadcrumbs getBreadcrumbs() {
        return breadcrumbs;
    }

    @InternalApi
    public static final class Builder {
        Session session;
        UserInfo userInfo;
        AppInfo appInfo;
        DeviceInfo deviceInfo;
        SessionPerformanceInfo performanceInfo;
        Breadcrumbs breadcrumbs;

        Builder() {
        }

        @NonNull
        public Builder withSession(@NonNull Session session) {
            this.session = session;
            return this;
        }

        @NonNull
        public Builder withUserInfo(@NonNull UserInfo userInfo) {
            this.userInfo = userInfo;
            return this;
        }

        @NonNull
        public Builder withAppInfo(@NonNull AppInfo appInfo) {
            this.appInfo = appInfo;
            return this;
        }

        @NonNull
        public Builder withDeviceInfo(@NonNull DeviceInfo deviceInfo) {
            this.deviceInfo = deviceInfo;
            return this;
        }

        @NonNull
        public Builder withPerformanceInfo(@Nullable SessionPerformanceInfo performanceInfo) {
            this.performanceInfo = performanceInfo;
            return this;
        }

        @NonNull
        public Builder withBreadcrumbs(Breadcrumbs breadcrumbs) {
            this.breadcrumbs = breadcrumbs;
            return this;
        }

        @NonNull
        public SessionMessage build() {
            return new SessionMessage(this);
        }
    }
}
