package io.embrace.android.embracesdk;

import androidx.annotation.Nullable;
import androidx.annotation.NonNull;

import com.google.gson.annotations.SerializedName;

/**
 * Represents performance info for a {@link Session}. This had to be sub-classed, as the API requires
 * the same key, 'nr', be used to represent two completely different objects. In order to overload
 * this key, we need two implementations of the {@link PerformanceInfo}.
 * <p>
 * For a {@link Session} this contains a network timeline.
 */
final class SessionPerformanceInfo extends PerformanceInfo {

    @SerializedName("nr")
    private final NetworkRequests networkRequests;

    SessionPerformanceInfo(Builder builder) {
        super(builder);
        this.networkRequests = builder.networkRequests;
    }

    public NetworkRequests getNetworkRequests() {
        return networkRequests;
    }

    public static SessionPerformanceInfo.Builder newBuilder() {
        return new SessionPerformanceInfo.Builder();
    }

    public static SessionPerformanceInfo.Builder fromPerformanceInfo(PerformanceInfo performanceInfo) {
        SessionPerformanceInfo.Builder builder = new SessionPerformanceInfo.Builder();

        if (performanceInfo == null) {
            return builder;
        }

        builder.withBatteryLevel(performanceInfo.getBatteryLevel());
        builder.withDiskUsage(performanceInfo.getDiskUsage());
        builder.withMemorySamples(performanceInfo.getMemorySamples());
        builder.withMemoryWarnings(performanceInfo.getMemoryWarnings());
        builder.withChargingIntervals(performanceInfo.getChargingIntervals());
        builder.withCriticalCpuIntervals(performanceInfo.getCriticalCpuIntervals());
        builder.withNetworkInterfaceIntervals(performanceInfo.getNetworkInterfaceIntervals());
        builder.withBatteryMeasurements(performanceInfo.getBatteryMeasurements());
        builder.withAnrIntervals(performanceInfo.getAnrIntervals());
        builder.withSignalStrength(performanceInfo.getSignalStrength());
        builder.withConnectionQualityIntervals(performanceInfo.getConnectionQualityIntervals());

        return builder;
    }

    static class Builder extends PerformanceInfo.Builder {

        NetworkRequests networkRequests;

        Builder() {
        }

        public Builder withNetworkRequests(@Nullable NetworkRequests NetworkRequests) {
            this.networkRequests = NetworkRequests;
            return this;
        }

        @NonNull
        public SessionPerformanceInfo build() {
            return new SessionPerformanceInfo(this);
        }
    }

    static class NetworkRequests {
        @SerializedName("v2")
        private final NetworkSessionV2 networkSessionV2;

        NetworkRequests(NetworkSessionV2 networkSessionV2) {
            this.networkSessionV2 = networkSessionV2;
        }
    }
}
