package io.embrace.android.embracesdk;

import androidx.annotation.Nullable;

import io.embrace.android.embracesdk.utils.optional.Optional;

import java.io.Closeable;
import java.util.Collection;
import java.util.Map;

interface SessionService extends Closeable {

    /**
     * Starts a new session.
     *
     * @param coldStart whether this is a cold start of the application
     * @param startType the origin of the event that starts the session.
     */
    void startSession(boolean coldStart, Session.SessionLifeEventType startType);

    /**
     * This is responsible for the current session to be cached, ended and sent to the server and
     * immediately starting a new session after that.
     *
     * @param endType the origin of the event that ends the session.
     */
    void triggerStatelessSessionEnd(Session.SessionLifeEventType endType);
    
    /**
     * Handles an uncaught exception, ending the session and saving the session to disk.
     */
    void handleCrash(String crashId);

    /**
     * Handles an native code uncaught exception, ending the session and saving the session to disk.
     */
    void handleNativeCrash(Optional<NativeCrashData> nativeCrash);

    /**
     * Annotates the session with a new property.  Use this to track permanent and ephemeral
     * features of the session. A permanent property is added to all sessions submitted from this
     * device, use this for properties such as work site, building, owner. A non-permanent property
     * is added to only the currently active session.
     * <p>
     * There is a maximum of 10 total properties in a session.
     *
     * @param key       The key for this property, must be unique within session properties
     * @param value     The value to store for this property
     * @param permanent If true the property is applied to all sessions going forward, persist
     *                  through app launches.
     * @return A boolean indicating whether the property was added or not
     */
    boolean addProperty(String key, String value, boolean permanent);

    /**
     * Removes a property from the session.  If that property was permanent then it is removed from
     * all future sessions as well.
     *
     * @param key the key to be removed
     */
    boolean removeProperty(String key);

    /**
     * Get a read-only representation of the currently set session properties.  You can query and
     * read from this representation however setting values in this object will not modify the
     * actual properties in the session. To modify session properties see addProperty.
     */
    @Nullable
    Map<String, String> getProperties();

    /**
     * Adds a listener which is invoked when a session start message is constructed.
     * You should use this to alter the session message rather than changing code in the
     * SessionService directly.
     *
     * @param listener the custom listener
     */
    void addSessionStartListeners(Collection<SessionStartListener> listener);

    /**
     * Adds a listener which is invoked when a session end message is constructed.
     * You should use this to alter the session message rather than changing code in the
     * SessionService directly.
     *
     * @param listener the custom listener
     */
    void addSessionEndListeners(Collection<SessionEndListener> listener);
}
