package io.embrace.android.embracesdk;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.optional.Optional;
import com.google.gson.annotations.SerializedName;

import java.lang.reflect.Method;

/**
 * Describes the cellular signal strength of the device, across all cellular connection types.
 */
final class SignalStrength {
    /** The timestamp at which the measurement was taken.*/
    @SerializedName("ts")
    private final long timestamp;

    /** The CDMA RSSI value in dBm. */
    @SerializedName("crs")
    private final Integer cdmaDbm;

    /** The CDMA Ec/Io value in dB*10. */
    @SerializedName("cec")
    private final Integer cdmaEcio;

    /** The EVDO Ec/Io value in dB*10. */
    @SerializedName("eec")
    private final Integer evdoEcio;

    /** The EVDO signal to noise ratio. */
    @SerializedName("esn")
    private final Integer evdoSnr;

    /** Get the GSM bit error rate (0-7, 99) as defined in TS 27.007 8.5. */
    @SerializedName("gbe")
    private final Integer gsmBitErrorRate;

    /** GSM Signal Strength, valid values are (0-31, 99) as defined in TS 27.007 8.5. */
    @SerializedName("gss")
    private final Integer gsmSignalStrength;

    /** True if the connection is GSM, false otherwise. */
    @SerializedName("gsm")
    private final boolean gsm;

    /** The LTE signal strength. */
    @SerializedName("lss")
    private final Integer lteSignalStrength;

    /** LTE reference signal received power. */
    @SerializedName("lrp")
    private final Integer lteRsrp;

    /** LTE reference signal received quality. */
    @SerializedName("lrq")
    private final Integer lteRsrq;

    /** LTE reference signal signal-to-noise ratio. */
    @SerializedName("lsn")
    private final Integer lteRssnr;

    /** LTE channel quality indicator. */
    @SerializedName("lcq")
    private final Integer lteCqi;

    /**
     * Gets the timestamp of the measurement.
     *
     * @return the timestamp
     */
    public long getTimestamp() {
        return timestamp;
    }

    public Optional<Integer> getCdmaDbm() {
        return Optional.fromNullable(cdmaDbm);
    }

    public Optional<Integer> getCdmaEcio() {
        return Optional.fromNullable(cdmaEcio);
    }

    public Optional<Integer> getEvdoEcio() {
        return Optional.fromNullable(evdoEcio);
    }

    public Optional<Integer> getEvdoSnr() {
        return Optional.fromNullable(evdoSnr);
    }

    public Optional<Integer> getGsmBitErrorRate() {
        return Optional.fromNullable(gsmBitErrorRate);
    }

    public Optional<Integer> getGsmSignalStrength() {
        return Optional.fromNullable(gsmSignalStrength);
    }

    public boolean isGsm() {
        return gsm;
    }

    public Optional<Integer> getLteSignalStrength() {
        return Optional.fromNullable(lteSignalStrength);
    }

    public Optional<Integer> getLteRsrp() {
        return Optional.fromNullable(lteRsrp);
    }

    public Optional<Integer> getLteRsrq() {
        return Optional.fromNullable(lteRsrq);
    }

    public Optional<Integer> getLteRssnr() {
        return Optional.fromNullable(lteRssnr);
    }

    public Optional<Integer> getLteCqi() {
        return Optional.fromNullable(lteCqi);
    }

    private SignalStrength(Long timestamp, Integer cdmaDbm, Integer cdmaEcio, Integer evdoEcio, Integer evdoSnr, Integer gsmBitErrorRate, Integer gsmSignalStrength, boolean gsm, Integer lteSignalStrength, Integer lteRsrp, Integer lteRsrq, Integer lteRssnr, Integer lteCqi) {
        this.timestamp = timestamp;
        this.cdmaDbm = cdmaDbm;
        this.cdmaEcio = cdmaEcio;
        this.evdoEcio = evdoEcio;
        this.evdoSnr = evdoSnr;
        this.gsmBitErrorRate = gsmBitErrorRate;
        this.gsmSignalStrength = gsmSignalStrength;
        this.gsm = gsm;
        this.lteSignalStrength = lteSignalStrength;
        this.lteRsrp = lteRsrp;
        this.lteRsrq = lteRsrq;
        this.lteRssnr = lteRssnr;
        this.lteCqi = lteCqi;

    }

    static SignalStrength of(android.telephony.SignalStrength signalStrength) {
        Integer lteSignalStrength = null;
        Integer lteRsrp = null;
        Integer lteRsrq = null;
        Integer lteRssnr = null;
        Integer lteCqi = null;
        try {
            Method[] methods = android.telephony.SignalStrength.class
                    .getMethods();

            for (Method mthd : methods) {
                // Access LTE values by reflection if present
                switch(mthd.getName()) {
                    case "getLteSignalStrength":
                        lteSignalStrength = (Integer) mthd.invoke(signalStrength);
                        break;
                    case "getLteRsrp":
                        lteRsrp = (Integer) mthd.invoke(signalStrength);
                        break;
                    case "getLteRsrq":
                        lteRsrq = (Integer) mthd.invoke(signalStrength);
                        break;
                    case "getLteRssnr":
                        lteRssnr = (Integer) mthd.invoke(signalStrength);
                        break;
                    case "getLteCqi":
                        lteCqi = (Integer) mthd.invoke(signalStrength);
                        break;
                }
            }
        } catch (Exception ex) {
            InternalStaticEmbraceLogger.logDebug("Failed to access LTE signal strength values", ex);
        }
        return new SignalStrength(
                System.currentTimeMillis(),
                signalStrength.getCdmaDbm(),
                signalStrength.getCdmaEcio(),
                signalStrength.getEvdoEcio(),
                signalStrength.getEvdoSnr(),
                signalStrength.getGsmBitErrorRate(),
                signalStrength.getGsmSignalStrength(),
                signalStrength.isGsm(),
                lteSignalStrength,
                lteRsrp,
                lteRsrq,
                lteRssnr,
                lteCqi);

    }
}
