package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.annotations.SerializedName;

import java.util.HashSet;
import java.util.Set;

import io.embrace.android.embracesdk.utils.optional.Optional;

/**
 * Information about the user of the app, provided by the developer performing the integration.
 */
final class UserInfo {

    static final String PERSONA_NEW_USER = "new_user";
    static final String PERSONA_POWER_USER = "power_user";
    static final String PERSONA_LOGGED_IN = "logged_in";
    static final String PERSONA_VIP = "vip";
    static final String PERSONA_CREATOR = "content_creator";
    static final String PERSONA_TESTER = "tester";
    static final String PERSONA_PAYER = "payer";
    static final String PERSONA_FIRST_DAY_USER = "first_day";

    UserInfo(Builder builder) {
        userId = builder.userId;
        email = builder.email;
        username = builder.username;
        personas = builder.personas;
    }

    public String toJson() {
        return "{\"id\": " + MessageUtils.withNull(userId) +
                ",\"em\": " + MessageUtils.withNull(email) +
                ",\"un\":" + MessageUtils.withNull(username) +
                ",\"per\":" + MessageUtils.withSet(personas) + "}";
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(UserInfo copy) {
        Builder builder = new Builder();

        if (copy == null) {
            return builder;
        }

        builder.userId = copy.getUserId();
        builder.email = copy.getEmail();
        builder.username = copy.getUsername();
        builder.personas = copy.getPersonas();
        return builder;
    }

    @SerializedName("id")
    private String userId;

    @SerializedName("em")
    private String email;

    @SerializedName("un")
    private String username;

    @SerializedName("per")
    private Set<String> personas;

    UserInfo(String userId, String email, String username, Set<String> personas) {
        this.userId = userId;
        this.email = email;
        this.username = username;
        this.personas = personas;
    }

    @Nullable
    public String getUserId() {
        return userId;
    }

    public @Nullable
    String getEmail() {
        return email;
    }

    public @Nullable
    String getUsername() {
        return username;
    }

    public @Nullable
    Set<String> getPersonas() {
        return personas;
    }

    void setUserId(String userId) {
        this.userId = userId;
    }

    void setEmail(String email) {
        this.email = email;
    }

    void setUsername(String username) {
        this.username = username;
    }

    void setPersonas(Set<String> personas) {
        this.personas = personas;
    }

    /**
     * Creates an empty instance.
     *
     * @return an empty instance
     */
    public static UserInfo ofEmpty() {
        return new UserInfo(null, null, null, null);
    }

    /**
     * Creates an instance of {@link UserInfo} from the cache.
     *
     * @param preferencesService the preferences service
     * @return user info created from the cache and configuration
     */
    public static UserInfo ofStored(PreferencesService preferencesService) {
        Builder builder = newBuilder();
        Set<String> personas = new HashSet<>();
        Optional<String> userEmailAddress = preferencesService.getUserEmailAddress();
        if (userEmailAddress.isPresent()) {
            builder.withEmail(userEmailAddress.get());
        }

        Optional<String> userIdentifier = preferencesService.getUserIdentifier();
        if (userIdentifier.isPresent()) {
            builder.withUserId(userIdentifier.get());
        }

        Optional<String> username = preferencesService.getUsername();
        if (username.isPresent()) {
            builder.withUsername(username.get());
        }

        Optional<Set<String>> userPersonas = preferencesService.getUserPersonas();
        if (userPersonas.isPresent()) {
            personas.addAll(userPersonas.get());
        }

        Optional<Set<String>> customPersonas = preferencesService.getCustomPersonas();
        if (customPersonas.isPresent()) {
            personas.addAll(customPersonas.get());
        }
        personas.remove(PERSONA_PAYER);
        if (preferencesService.getUserPayer()) {
            personas.add(PERSONA_PAYER);
        }
        personas.remove(PERSONA_FIRST_DAY_USER);
        if (preferencesService.isUsersFirstDay()) {
            personas.add(PERSONA_FIRST_DAY_USER);
        }

        builder.withPersonas(personas);
        return builder.build();
    }

    public static final class Builder {
        String userId;
        String email;
        String username;
        Set<String> personas;

        Builder() {
        }

        @NonNull
        public Builder withUserId(String userId) {
            this.userId = userId;
            return this;
        }

        @NonNull
        public Builder withEmail(String email) {
            this.email = email;
            return this;
        }

        @NonNull
        public Builder withUsername(String username) {
            this.username = username;
            return this;
        }

        @NonNull
        public Builder withPersonas(@Nullable Set<String> personas) {
            this.personas = personas;
            return this;
        }

        @Nullable
        public String getUserId() {
            return userId;
        }

        @Nullable
        public String getEmail() {
            return email;
        }

        @Nullable
        public String getUsername() {
            return username;
        }

        @Nullable
        public Set<String> getPersonas() {
            return personas;
        }

        @NonNull
        public UserInfo build() {
            return new UserInfo(this);
        }
    }
}