package io.embrace.android.embracesdk.config

import com.google.gson.annotations.SerializedName
import java.util.regex.Pattern

/**
 * Configuration values relating to the ANR tracking on the app.
 */
internal data class AnrConfig @JvmOverloads constructor(

    @field:Transient
    private val timestamp: Long? = null,

    @SerializedName(PCT_ENABLED)
    private val pctEnabled: Int? = null,

    @SerializedName(PCT_BG_ENABLED)
    private val pctBgEnabled: Int? = null,

    @SerializedName(INTERVAL)
    private val intervalMs: Long? = null,

    @SerializedName(PER_INTERVAL)
    private val stacktracesPerInterval: Int? = null,

    @SerializedName(MAX_DEPTH)
    private val stacktraceMaxDepth: Int? = null,

    @SerializedName(PER_SESSION)
    private val anrPerSession: Int? = null,

    @SerializedName(MAIN_THREAD_ONLY)
    private val mainThreadOnly: Boolean? = null,

    @SerializedName(ALLOW_EARLY_CAPTURE)
    private val allowEarlyCapture: Boolean? = null,

    @SerializedName(PRIORITY)
    private val priority: Int? = null,

    @SerializedName(MIN_DURATION)
    private val minDuration: Int? = null,

    @SerializedName(WHITE_LIST)
    private val allowList: List<String>? = emptyList(),

    @SerializedName(BLACK_LIST)
    private val blockList: List<String>? = emptyList(),

    @SerializedName(UNITY_NDK_SAMPLING_FACTOR)
    private val unityNdkSamplingFactor: Int? = null,

    @SerializedName(UNITY_NDK_SAMPLING_UNWINDER)
    private val unityNdkSamplingUnwinder: String? = DEFAULT_UNITY_NDK_SAMPLING_UNWINDER,

    @SerializedName(PCT_UNITY_NDK_SAMPLING_ENABLED)
    private val pctUnityNdkSamplingEnabled: Float? = null,

    @SerializedName(NDK_SAMPLING_OFFSET_ENABLED)
    private val ndkSamplingOffsetEnabled: Boolean? = null,

    @SerializedName(IGNORE_UNITY_NDK_SAMPLING_ALLOWLIST)
    private val ignoreUnityNdkSamplingAllowlist: Boolean? = null,

    @SerializedName(UNITY_NDK_SAMPLING_ALLOWLIST)
    private val unityNdkSamplingAllowlist: List<AllowedNdkSampleMethod>? = null,

    /**
     * Percentage of users for which Google ANR timestamp capture is enabled.
     */
    @SerializedName(GOOGLE_PCT_ENABLED)
    private val googlePctEnabled: Int? = null
) {

    enum class Unwinder(internal val code: Int) {
        LIBUNWIND(0),
        LIBUNWINDSTACK(1);
    }

    /**
     * Allow listed threads by pattern
     */
    @delegate:Transient
    val allowPatternList: List<Pattern> by lazy {
        allowList?.map(Pattern::compile) ?: emptyList()
    }

    /**
     * Black listed threads by pattern
     */
    @delegate:Transient
    val blockPatternList: List<Pattern> by lazy {
        blockList?.map(Pattern::compile) ?: emptyList()
    }

    /**
     * The timestamp at which this data was fetched from the server.
     */
    fun getTimestamp(): Long = timestamp ?: DEFAULT_ANR_TIMESTAMP

    /**
     * Percentage of users for which ANR stack trace capture is enabled.
     */
    fun getPctEnabled(): Int = pctEnabled ?: DEFAULT_ANR_PCT_ENABLED

    /**
     * Percentage of users for which Google ANR stack trace capture is enabled.
     */
    fun getGooglePctEnabled(): Int = googlePctEnabled ?: DEFAULT_ANR_GOOGLE_PCT_ENABLED

    /**
     * Percentage of users for which Background ANR stack trace capture is enabled.
     */
    fun getPctBgEnabled(): Int = pctBgEnabled ?: DEFAULT_ANR_BG_PCT_ENABLED

    /**
     * Time between stack trace captures for time intervals after the start of an ANR.
     */
    fun getIntervalMs(): Long = intervalMs ?: DEFAULT_ANR_INTERVAL_MS

    /**
     * Maximum captured stacktraces for a single ANR interval.
     */
    fun getStacktracesPerInterval(): Int = stacktracesPerInterval ?: DEFAULT_ANR_MAX_PER_INTERVAL

    /**
     * Maximum depth of stacktrace to keep.
     */
    fun getStacktraceMaxLength(): Int = stacktraceMaxDepth ?: DEFAULT_ANR_MAX_STACK_DEPTH

    /**
     * Maximum captured anr for a session.
     */
    fun getMaxAnrCapturedIntervalsPerSession(): Int =
        anrPerSession ?: DEFAULT_ANR_MAX_ANR_INTERVALS_PER_SESSION

    /**
     * The min thread priority that should be captured
     */
    fun getPriority(): Int = priority ?: DEFAULT_ANR_MIN_THREAD_PRIORITY_TO_CAPTURE

    /**
     * Minimum duration of an ANR interval
     */
    fun getMinDuration(): Int = minDuration ?: DEFAULT_ANR_MIN_CAPTURE_DURATION

    /**
     * Whether only the main thread should be captured
     */
    fun isMainThreadOnly(): Boolean = mainThreadOnly ?: DEFAULT_ANR_MAIN_THREAD_ONLY

    /**
     * Whether early ANR capture should be enabled
     */
    fun isAllowEarlyCapture(): Boolean = allowEarlyCapture ?: DEFAULT_ALLOW_EARLY_CAPTURE

    /**
     * Allow-listed threads by name.
     */
    fun getAllowList(): List<String> = allowList ?: emptyList()

    /**
     * Black-listed threads by name.
     */
    fun getBlockList(): List<String> = blockList ?: emptyList()

    /**
     * The sampling factor for NDK stacktrace sampling. This should be multiplied by
     * the [intervalMs] to give the NDK sampling interval.
     */
    fun getUnityNdkSamplingFactor(): Int =
        unityNdkSamplingFactor ?: DEFAULT_UNITY_NDK_SAMPLING_FACTOR

    /**
     * The unwinder used for NDK stacktrace sampling.
     */
    fun getUnityNdkSamplingUnwinder(): Unwinder {
        return runCatching {
            Unwinder.values().find {
                it.name.equals(unityNdkSamplingUnwinder, true)
            } ?: Unwinder.LIBUNWIND
        }.getOrDefault(Unwinder.LIBUNWIND)
    }

    /**
     * The percentage of enabled devices for NDK stacktrace sampling.
     */
    fun getPctUnityNdkSamplingEnabled(): Float =
        pctUnityNdkSamplingEnabled ?: DEFAULT_PCT_UNITY_NDK_SAMPLING_ENABLED

    /**
     * Whether offsets are enabled for NDK stacktrace sampling.
     */
    fun getNdkSamplingOffsetEnabled(): Boolean =
        ndkSamplingOffsetEnabled ?: DEFAULT_NDK_SAMPLING_OFFSET_ENABLED

    /**
     * Whether the allow list is ignored or not.
     */
    fun getIgnoreUnityNdkSamplingAllowlist(): Boolean =
        ignoreUnityNdkSamplingAllowlist ?: DEFAULT_IGNORE_UNITY_NDK_SAMPLING_ALLOWLIST

    /**
     * The allowed list of classes/methods for NDK stacktrace sampling
     */
    fun getUnityNdkSamplingAllowlist(): List<AllowedNdkSampleMethod> =
        unityNdkSamplingAllowlist ?: DEFAULT_UNITY_NDK_SAMPLING_ALLOWLIST

    /**
     * The sampling factor for NDK stacktrace sampling. This is calculated by multiplying
     * the [intervalMs] against [unityNdkSamplingFactor].
     */
    fun getUnityNdkSamplingIntervalMs() = getIntervalMs() * getUnityNdkSamplingFactor()

    companion object {
        const val DEFAULT_ANR_TIMESTAMP: Long = -1
        const val DEFAULT_ANR_PCT_ENABLED = 100
        const val DEFAULT_ANR_BG_PCT_ENABLED = 0
        const val DEFAULT_ANR_INTERVAL_MS: Long = 100
        const val DEFAULT_ANR_MAX_PER_INTERVAL = 80
        const val DEFAULT_ANR_MAX_STACK_DEPTH = 100
        const val DEFAULT_ANR_MIN_THREAD_PRIORITY_TO_CAPTURE = 0
        const val DEFAULT_ANR_MAX_ANR_INTERVALS_PER_SESSION = 3
        const val DEFAULT_ANR_MIN_CAPTURE_DURATION = 1000
        const val DEFAULT_ANR_MAIN_THREAD_ONLY = true
        const val DEFAULT_ALLOW_EARLY_CAPTURE = true
        const val DEFAULT_UNITY_NDK_SAMPLING_FACTOR = 5
        const val DEFAULT_UNITY_NDK_SAMPLING_UNWINDER = "libunwind"
        const val DEFAULT_PCT_UNITY_NDK_SAMPLING_ENABLED = 0.00f
        const val DEFAULT_NDK_SAMPLING_OFFSET_ENABLED = true
        const val DEFAULT_IGNORE_UNITY_NDK_SAMPLING_ALLOWLIST = true
        val DEFAULT_UNITY_NDK_SAMPLING_ALLOWLIST = listOf(
            AllowedNdkSampleMethod("UnityPlayer", "pauseUnity")
        )
        // Change to 100 once feature is out of beta
        const val DEFAULT_ANR_GOOGLE_PCT_ENABLED = 0

        const val ANR_CFG_TIMESTAMP = "timestamp"
        const val PCT_ENABLED = "pct_enabled"
        const val GOOGLE_PCT_ENABLED = "google_pct_enabled"
        const val PCT_BG_ENABLED = "pct_bg_enabled"
        const val INTERVAL = "interval"
        const val PER_INTERVAL = "per_interval"
        const val MAX_DEPTH = "max_depth"
        const val PER_SESSION = "per_session"
        const val MAIN_THREAD_ONLY = "main_thread_only"
        const val PRIORITY = "priority"
        const val MIN_DURATION = "min_duration"
        const val WHITE_LIST = "white_list"
        const val BLACK_LIST = "black_list"
        const val ALLOW_EARLY_CAPTURE = "allow_early_capture"
        const val UNITY_NDK_SAMPLING_FACTOR = "unity_ndk_sampling_factor"
        const val UNITY_NDK_SAMPLING_UNWINDER = "unity_ndk_sampling_unwinder"
        const val PCT_UNITY_NDK_SAMPLING_ENABLED = "pct_unity_ndk_sampling_enabled"
        const val NDK_SAMPLING_OFFSET_ENABLED = "ndk_sampling_offset_enabled"
        const val IGNORE_UNITY_NDK_SAMPLING_ALLOWLIST = "ignore_unity_ndk_sampling_allowlist"
        const val UNITY_NDK_SAMPLING_ALLOWLIST = "unity_ndk_sampling_allowlist"

        @JvmStatic
        fun ofDefault() = AnrConfig()
    }

    internal class AllowedNdkSampleMethod(
        @SerializedName("c") val clz: String? = null,
        @SerializedName("m") val method: String? = null
    )
}
