package io.embrace.android.embracesdk.config

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger
import java.util.*

/**
 * Represents the session components that should be added to the payload if the feature gating is
 * enabled by config.
 * Also defines events (like crashes or error logs) that must send full session payloads.
 */
internal data class GatingConfig(
    val components: Set<String>?,
    var fullSessionEvents: Set<String>?
) {

    init {
        InternalStaticEmbraceLogger.logDeveloper("GatingConfig", "Initializing GatingConfig")
        fullSessionEvents = sessionEventsToLowerCase()
        InternalStaticEmbraceLogger.logDeveloper(
            "GatingConfig",
            "Sending full session for events: ${fullSessionEvents?.joinToString("-") ?: " empty "}"
        )

        InternalStaticEmbraceLogger.logDeveloper(
            "GatingConfig",
            "Components to add: ${components?.joinToString("-")}"
        )
    }

    private fun sessionEventsToLowerCase() =
        fullSessionEvents?.map { it.toLowerCase(Locale.getDefault()) }?.toSet()

    companion object {
        const val BREADCRUMBS_TAPS = "br_tb"
        const val BREADCRUMBS_VIEWS = "br_vb"
        const val BREADCRUMBS_CUSTOM_VIEWS = "br_cv"
        const val BREADCRUMBS_WEB_VIEWS = "br_wv"
        const val BREADCRUMBS_CUSTOM = "br_cb"
        const val LOG_PROPERTIES = "log_pr"
        const val SESSION_PROPERTIES = "s_props"
        const val SESSION_ORIENTATIONS = "s_oc"
        const val SESSION_USER_TERMINATION = "s_tr"
        const val SESSION_MOMENTS = "s_mts"
        const val LOGS_INFO = "log_in"
        const val LOGS_WARN = "log_war"
        const val STARTUP_MOMENT = "mts_st"
        const val USER_PERSONAS = "ur_per"
        const val PERFORMANCE_ANR = "pr_anr"
        const val PERFORMANCE_CONNECTIVITY = "pr_ns"
        const val PERFORMANCE_NETWORK = "pr_nr"
        const val PERFORMANCE_CPU = "pr_cp"
        const val PERFORMANCE_LOW_MEMORY = "pr_mw"
        const val PERFORMANCE_MEMORY_SAMPLES = "pr_me"
        const val PERFORMANCE_CHARGING_INTERVALS = "pr_bc"
        const val PERFORMANCE_BATTERY_MEASUREMENTS = "pr_bm"
        const val PERFORMANCE_BATTERY_LEVEL = "pr_bt"
        const val PERFORMANCE_CURRENT_DISK_USAGE = "pr_ds"

        // Events that can send full session payloads
        const val FULL_SESSION_CRASHES = "crashes"
        const val FULL_SESSION_ERROR_LOGS = "errors"
    }
}
