package io.embrace.android.embracesdk.logging

import android.util.Log
import io.embrace.android.embracesdk.EmbraceLogger.Severity
import io.embrace.android.embracesdk.internal.ApkToolsConfig
import java.util.concurrent.CopyOnWriteArrayList

/**
 * Wrapper for the Android [Log] utility.
 * Can only be used internally, it's not part of the public API.
 */
internal class InternalEmbraceLogger {

    private val loggerActions: CopyOnWriteArrayList<LoggerAction> = CopyOnWriteArrayList<LoggerAction>()

    private var threshold = Severity.INFO

    internal interface LoggerAction {
        fun log(msg: String, severity: Severity, throwable: Throwable?, logStacktrace: Boolean)
    }

    fun addLoggerAction(action: LoggerAction) {
        loggerActions.add(action)
    }

    @JvmOverloads
    fun logDeveloper(className: String, msg: String, throwable: Throwable? = null) {
        log("[$className] $msg", Severity.DEVELOPER, throwable, true)
    }

    @JvmOverloads
    fun logDebug(msg: String, throwable: Throwable? = null) {
        log(msg, Severity.DEBUG, throwable, true)
    }

    fun logInfo(msg: String) {
        log(msg, Severity.INFO, null, true)
    }

    @JvmOverloads
    fun logWarning(msg: String, throwable: Throwable? = null) {
        log(msg, Severity.WARNING, throwable, true)
    }

    @JvmOverloads
    fun logError(msg: String, throwable: Throwable? = null, logStacktrace: Boolean = false) {
        log(msg, Severity.ERROR, throwable, logStacktrace)
    }

    @JvmOverloads
    fun logSDKNotInitialized(action: String) {
        log("Embrace SDK is not initialized yet, cannot $action.", Severity.ERROR, Throwable(), true)
    }

    /**
     * Logs a message.
     *
     * @param msg the message to log.
     * @param severity how severe the log is. If it's lower than the threshold, the message will not be logged.
     * @param throwable exception, if any.
     * @param logStacktrace should add the throwable to the logging
     */

    fun log(msg: String, severity: Severity, throwable: Throwable?, logStacktrace: Boolean) {
        if (shouldTriggerLoggerActions(severity))
            loggerActions.forEach {
                it.log(msg, severity, throwable, logStacktrace)
            }
    }

    private fun shouldTriggerLoggerActions(severity: Severity) =
        ApkToolsConfig.IS_DEVELOPER_LOGGING_ENABLED || severity >= threshold

    fun setThreshold(severity: Severity) {
        threshold = severity
    }
}
