package io.embrace.android.embracesdk.network;

import android.net.Uri;
import android.webkit.URLUtil;

import java.util.EnumSet;
import java.util.Locale;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.network.http.HttpMethod;

/**
 * This class is used to create manually-recorded network requests.
 */
public class EmbraceNetworkRequestV2 {

    private static final EnumSet<HttpMethod> allowedMethods = EnumSet.of(
            HttpMethod.GET,
            HttpMethod.PUT,
            HttpMethod.POST,
            HttpMethod.DELETE,
            HttpMethod.PATCH);

    /**
     * The request's URL. Must start with http:// or https://
     */
    private final String url;
    /**
     * The request's method. Must be one of the following: GET, PUT, POST, DELETE, PATCH.
     */
    private final HttpMethod httpMethod;
    /**
     * The time the request started.
     */
    private final Long startTime;
    /**
     * The time the request ended. Must be greater than the startTime.
     */
    private final Long endTime;
    /**
     * The number of bytes received.
     */
    private final Long bytesIn;
    /**
     * The number of bytes received.
     */
    private final Long bytesOut;
    /**
     * The response status of the request. Must be in the range 100 to 599.
     */
    private final Integer responseCode;
    /**
     * Error that describes a non-HTTP error, e.g. a connection error.
     */
    private final Throwable error;
    /**
     * Optional trace ID that can be used to trace a particular request. Max length is 64 characters.
     */
    private final String traceId;

    EmbraceNetworkRequestV2(Builder builder) {
        this.url = builder.urlString;
        this.httpMethod = builder.httpMethod;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.bytesIn = builder.bytesIn;
        this.bytesOut = builder.bytesOut;
        this.responseCode = builder.responseCode;
        this.error = builder.error;
        this.traceId = builder.traceId;
    }

    static boolean validateMethod(HttpMethod method) {

        if (method == null) {
            InternalStaticEmbraceLogger.logError("Method cannot be null");
            return false;
        }

        if (!allowedMethods.contains(method)) {
            InternalStaticEmbraceLogger.logError("Not a valid method: " + method.name());
            return false;
        }

        return true;
    }

    public boolean canSend() {
        if (this.url == null) {
            InternalStaticEmbraceLogger.logError("Request must contain URL");
            return false;
        }
        if (this.httpMethod == null) {
            InternalStaticEmbraceLogger.logError("Request must contain method");
            return false;
        }
        if (this.startTime == null) {
            InternalStaticEmbraceLogger.logError("Request must contain startTime");
            return false;
        }
        if (this.endTime == null) {
            InternalStaticEmbraceLogger.logError("Request must contain endTime");
            return false;
        }
        if ((this.responseCode == null || this.responseCode == -1) && this.error == null) {
            InternalStaticEmbraceLogger.logError("Request must either have responseCode or error set");
            return false;
        }

        return true;
    }

    @NonNull
    public String getUrl() {
        return url;
    }

    @NonNull
    public String getHttpMethod() {
        return httpMethod != null ? httpMethod.name().toUpperCase() : null;
    }

    @NonNull
    public Long getStartTime() {
        return startTime;
    }

    @Nullable
    public Long getEndTime() {
        return endTime;
    }

    @NonNull
    public Long getBytesIn() {
        return bytesIn == null ? 0 : bytesIn;
    }

    @NonNull
    public Long getBytesOut() {
        return bytesOut == null ? 0 : bytesOut;
    }

    @Nullable
    public Integer getResponseCode() {
        return responseCode;
    }

    @Nullable
    public Throwable getError() {
        return error;
    }

    @Nullable
    public String getTraceId() {
        return traceId;
    }

    @NonNull
    public String description() {
        return "<" + this + ": " + this +
                " URL = " + this.url +
                " Method = " + this.httpMethod +
                " Start = " + this.startTime + ">";
    }

    @NonNull
    public static Builder newBuilder() {
        return new Builder();
    }

    public static final class Builder {
        String urlString;
        HttpMethod httpMethod;
        Long startTime;
        Long endTime;
        Long bytesIn;
        Long bytesOut;
        Integer responseCode;
        Throwable error;
        String traceId;

        Builder() {
        }

        @NonNull
        public Builder withUrl(@NonNull String urlString) {
            this.urlString = urlString;
            return this;
        }

        @NonNull
        public Builder withHttpMethod(@NonNull HttpMethod httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        @NonNull
        public Builder withHttpMethod(int httpMethod) {
            this.httpMethod = HttpMethod.fromInt(httpMethod);
            return this;
        }

        @NonNull
        public Builder withStartTime(@NonNull Long startTime) {
            this.startTime = startTime;
            return this;
        }

        @NonNull
        public Builder withEndTime(@NonNull Long date) {

            if (date == null) {
                InternalStaticEmbraceLogger.logError("End time cannot be null");
                return this;
            }

            if (date <= this.startTime) {
                InternalStaticEmbraceLogger.logError("End time cannot be before the start time");
                return this;
            }

            this.endTime = date;
            return this;
        }

        @Deprecated
        @NonNull
        public Builder withByteIn(@NonNull Long bytesIn) {
            return withBytesIn(bytesIn);
        }

        @NonNull
        public Builder withBytesIn(int bytesIn) {
            return withBytesIn((long) bytesIn);
        }

        @NonNull
        public Builder withBytesIn(@NonNull Long bytesIn) {

            if (bytesIn == null) {
                InternalStaticEmbraceLogger.logError("BytesIn cannot be null");
                return this;
            }

            if (bytesIn < 0) {
                InternalStaticEmbraceLogger.logError("BytesIn must be a positive long");
                return this;
            }

            this.bytesIn = bytesIn;
            return this;
        }

        @Deprecated
        @NonNull
        public Builder withByteOut(@NonNull Long bytesOut) {
            return withBytesOut(bytesOut);
        }

        @NonNull
        public Builder withBytesOut(int bytesOut) {
            return withBytesOut((long) bytesOut);
        }

        @NonNull
        public Builder withBytesOut(@NonNull Long bytesOut) {

            if (bytesOut == null) {
                InternalStaticEmbraceLogger.logError("BytesOut cannot be null");
                return this;
            }

            if (bytesOut < 0) {
                InternalStaticEmbraceLogger.logError("BytesOut must be a positive long");
                return this;
            }

            this.bytesOut = bytesOut;
            return this;
        }

        @NonNull
        public Builder withResponseCode(@NonNull Integer responseCode) {

            if (responseCode < 100 || responseCode > 599) {
                InternalStaticEmbraceLogger.logError(String.format(Locale.getDefault(), "Invalid responseCode: %d", responseCode));
                return this;
            }

            this.responseCode = responseCode;
            return this;
        }

        @NonNull
        public Builder withError(@NonNull Throwable error) {

            if (error == null) {
                InternalStaticEmbraceLogger.logError("Ignoring null error");
                return this;
            }

            this.error = error;
            return this;
        }

        @NonNull
        public Builder withTraceId(@NonNull String traceId) {
            this.traceId = traceId;
            return this;
        }

        @NonNull
        public EmbraceNetworkRequestV2 build() {

            Uri url;
            try {
                url = Uri.parse(urlString);
            } catch (NullPointerException ex) {
                InternalStaticEmbraceLogger.logError("Failed to parse URL " + urlString);
                return null;
            }

            if (url == null || url.getScheme() == null || url.getHost() == null) {
                InternalStaticEmbraceLogger.logError("Invalid URL: " + urlString);
                return null;
            }

            if (!URLUtil.isHttpsUrl(urlString) && !URLUtil.isHttpUrl(urlString)) {
                InternalStaticEmbraceLogger.logError("Only http and https schemes are supported: " + urlString);
                return null;
            }

            if (!validateMethod(httpMethod)) {
                return null;
            }

            if (startTime == null) {
                InternalStaticEmbraceLogger.logError("Start time cannot be null");
                return null;
            }

            return new EmbraceNetworkRequestV2(this);
        }
    }
}
