package io.embrace.android.embracesdk.utils.exceptions;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import io.embrace.android.embracesdk.InternalApi;
import io.embrace.android.embracesdk.utils.Function;
import io.embrace.android.embracesdk.utils.exceptions.function.CheckedFunction;
import io.embrace.android.embracesdk.utils.exceptions.function.CheckedRunnable;
import io.embrace.android.embracesdk.utils.exceptions.function.CheckedSupplier;

/**
 * Static utility methods that convert checked exceptions to unchecked.
 * <p>
 * Two {@code wrap()} methods are provided that can wrap an arbitrary piece of logic
 * and convert checked exceptions to unchecked.
 * <p>
 * A number of other methods are provided that allow a lambda block to be decorated
 * to avoid handling checked exceptions.
 * For example, the method {@link File#getCanonicalFile()} throws an {@link IOException}
 * which can be handled as follows:
 * <pre>
 *  stream.map(Unchecked.function(file -&gt; file.getCanonicalFile())
 * </pre>
 * <p>
 * Each method accepts a functional interface that is defined to throw {@link Throwable}.
 * Catching {@code Throwable} means that any method can be wrapped.
 * Any {@code InvocationTargetException} is extracted and processed recursively.
 * Any {@link Error} or {@link RuntimeException} is re-thrown without alteration.
 * Any other exception is wrapped in a {@link RuntimeException}.
 */
@InternalApi
public final class Unchecked {

    /**
     * Restricted constructor.
     */
    private Unchecked() {
    }

    //-------------------------------------------------------------------------

    /**
     * Wraps a block of code, converting checked exceptions to unchecked.
     * <pre>
     *   Unchecked.wrap(() -&gt; {
     *     // any code that throws a checked exception
     *   }
     * </pre>
     * <p>
     * If a checked exception is thrown it is converted to a {@link RuntimeException}.
     *
     * @param <T> the type of the result
     * @param block  the code block to wrap
     * @return the result of invoking the block
     * @throws RuntimeException if an exception occurs
     */
    public static <T> T wrap(CheckedSupplier<T> block) {
        try {
            return block.get();
        } catch (Throwable ex) {
            throw propagate(ex);
        }
    }

    //-------------------------------------------------------------------------
    /**
     * Converts checked exceptions to unchecked based on the {@code Runnable} interface.
     * <p>
     * This wraps the specified runnable returning an instance that handles checked exceptions.
     * If a checked exception is thrown it is converted to a {@link RuntimeException}.
     *
     * @param runnable  the runnable to be decorated
     * @return the runnable instance that handles checked exceptions
     */
    public static Runnable runnable(CheckedRunnable runnable) {
        return () -> {
            try {
                runnable.run();
            } catch (Throwable ex) {
                throw propagate(ex);
            }
        };
    }

    //-------------------------------------------------------------------------
    /**
     * Converts checked exceptions to unchecked based on the {@code Function} interface.
     * <p>
     * This wraps the specified function returning an instance that handles checked exceptions.
     * If a checked exception is thrown it is converted to a {@link RuntimeException}.
     *
     * @param <T>  the input type of the function
     * @param <R>  the return type of the function
     * @param function  the function to be decorated
     * @return the function instance that handles checked exceptions
     */
    public static <T, R> Function<T, R> function(CheckedFunction<T, R> function) {
        return (t) -> {
            try {
                return function.apply(t);
            } catch (Throwable ex) {
                throw propagate(ex);
            }
        };
    }

    /**
     * Propagates {@code throwable} as-is if possible, or by wrapping in a {@code RuntimeException} if not.
     * <ul>
     *   <li>If {@code throwable} is an {@code InvocationTargetException} the cause is extracted and processed recursively.</li>
     *   <li>If {@code throwable} is an {@code InterruptedException} then the thread is interrupted and a {@code RuntimeException} is thrown.</li>
     *   <li>If {@code throwable} is an {@code Error} or {@code RuntimeException} it is propagated as-is.</li>
     *   <li>Otherwise {@code throwable} is wrapped in a {@code RuntimeException} and thrown.</li>
     * </ul>
     * This method always throws an exception. The return type is a convenience to satisfy the type system
     * when the enclosing method returns a value. For example:
     * <pre>
     *   T foo() {
     *     try {
     *       return methodWithCheckedException();
     *     } catch (Exception e) {
     *       throw Unchecked.propagate(e);
     *     }
     *   }
     * </pre>
     *
     * @param throwable the {@code Throwable} to propagate
     * @return nothing; this method always throws an exception
     */
    public static RuntimeException propagate(Throwable throwable) {
        if (throwable instanceof InvocationTargetException) {
            throw propagate(throwable.getCause());
        } else {
            if (throwable instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new RuntimeException(throwable);
        }
    }

}