/*
 * Decompiled with CFR 0.152.
 */
package io.evitadb.externalApi.graphql;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.linecorp.armeria.server.HttpService;
import graphql.GraphQL;
import graphql.schema.GraphQLSchema;
import io.evitadb.api.CatalogContract;
import io.evitadb.core.CorruptedCatalog;
import io.evitadb.core.Evita;
import io.evitadb.exception.EvitaInternalError;
import io.evitadb.externalApi.graphql.api.catalog.CatalogGraphQLBuilder;
import io.evitadb.externalApi.graphql.api.catalog.dataApi.CatalogDataApiGraphQLSchemaBuilder;
import io.evitadb.externalApi.graphql.api.catalog.schemaApi.CatalogSchemaApiGraphQLSchemaBuilder;
import io.evitadb.externalApi.graphql.api.system.SystemGraphQLBuilder;
import io.evitadb.externalApi.graphql.api.system.builder.SystemGraphQLSchemaBuilder;
import io.evitadb.externalApi.graphql.configuration.GraphQLConfig;
import io.evitadb.externalApi.graphql.exception.GraphQLInternalError;
import io.evitadb.externalApi.graphql.io.GraphQLInstanceType;
import io.evitadb.externalApi.graphql.io.GraphQLRouter;
import io.evitadb.externalApi.graphql.metric.event.instance.BuiltEvent;
import io.evitadb.externalApi.graphql.utils.GraphQLSchemaPrinter;
import io.evitadb.externalApi.http.PathNormalizingHandler;
import io.evitadb.utils.Assert;
import io.evitadb.utils.CollectionUtils;
import io.evitadb.utils.StringUtils;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GraphQLManager {
    private static final Logger log = LoggerFactory.getLogger(GraphQLManager.class);
    @Nonnull
    private final Evita evita;
    @Nonnull
    private final GraphQLConfig graphQLConfig;
    @Nonnull
    private final GraphQLRouter graphQLRouter;
    @Nonnull
    private final Set<String> registeredCatalogs = CollectionUtils.createHashSet((int)20);
    @Nullable
    private SystemBuildStatistics systemBuildStatistics;
    @Nonnull
    private final Map<String, CatalogBuildStatistics> catalogBuildStatistics = CollectionUtils.createHashMap((int)20);

    public GraphQLManager(@Nonnull Evita evita, @Nonnull GraphQLConfig graphQLConfig) {
        this.evita = evita;
        this.graphQLConfig = graphQLConfig;
        ObjectMapper objectMapper = new ObjectMapper();
        this.graphQLRouter = new GraphQLRouter(objectMapper, evita, graphQLConfig);
        long buildingStartTime = System.currentTimeMillis();
        this.registerSystemApi();
        this.evita.getCatalogs().forEach(catalog -> this.registerCatalog(catalog.getName()));
        log.info("Built GraphQL API in " + StringUtils.formatPreciseNano((long)(System.currentTimeMillis() - buildingStartTime)));
    }

    @Nonnull
    public HttpService getGraphQLRouter() {
        return new PathNormalizingHandler((HttpService)this.graphQLRouter);
    }

    private void registerSystemApi() {
        long instanceBuildStartTime = System.currentTimeMillis();
        long schemaBuildStartTime = System.currentTimeMillis();
        GraphQLSchema schema = new SystemGraphQLSchemaBuilder(this.graphQLConfig, this.evita).build();
        long schemaBuildDuration = System.currentTimeMillis() - schemaBuildStartTime;
        this.graphQLRouter.registerSystemApi(new SystemGraphQLBuilder(this.evita, schema).build(this.graphQLConfig));
        long instanceBuildDuration = System.currentTimeMillis() - instanceBuildStartTime;
        this.systemBuildStatistics = SystemBuildStatistics.createNew(instanceBuildDuration, schemaBuildDuration, GraphQLManager.countGraphQLSchemaLines(schema));
    }

    public void registerCatalog(@Nonnull String catalogName) {
        CatalogContract catalog = this.evita.getCatalogInstanceOrThrowException(catalogName);
        if (catalog instanceof CorruptedCatalog) {
            log.warn("Catalog `" + catalogName + "` is corrupted. Skipping...");
            return;
        }
        Assert.isPremiseValid((!this.registeredCatalogs.contains(catalogName) ? 1 : 0) != 0, () -> new GraphQLInternalError("Catalog `" + catalogName + "` has been already registered."));
        try {
            long dataApiInstanceBuildStartTime = System.currentTimeMillis();
            long dataApiSchemaBuildStartTime = System.currentTimeMillis();
            GraphQLSchema dataApiSchema = new CatalogDataApiGraphQLSchemaBuilder(this.graphQLConfig, this.evita, catalog).build();
            long dataApiSchemaBuildDuration = System.currentTimeMillis() - dataApiSchemaBuildStartTime;
            GraphQL dataApi = new CatalogGraphQLBuilder(this.evita, catalog, dataApiSchema).build(this.graphQLConfig);
            this.graphQLRouter.registerCatalogApi(catalogName, GraphQLInstanceType.DATA, dataApi);
            long dataApiInstanceBuildDuration = System.currentTimeMillis() - dataApiInstanceBuildStartTime;
            long schemaApiInstanceBuildStartTime = System.currentTimeMillis();
            long schemaApiSchemaBuildStartTime = System.currentTimeMillis();
            GraphQLSchema schemaApiSchema = new CatalogSchemaApiGraphQLSchemaBuilder(this.graphQLConfig, this.evita, catalog).build();
            long schemaApiSchemaBuildDuration = System.currentTimeMillis() - schemaApiSchemaBuildStartTime;
            GraphQL schemaApi = new CatalogGraphQLBuilder(this.evita, catalog, schemaApiSchema).build(this.graphQLConfig);
            this.graphQLRouter.registerCatalogApi(catalogName, GraphQLInstanceType.SCHEMA, schemaApi);
            long schemaApiInstanceBuildDuration = System.currentTimeMillis() - schemaApiInstanceBuildStartTime;
            this.registeredCatalogs.add(catalogName);
            CatalogBuildStatistics schemaBuildStatistics = CatalogBuildStatistics.createNew(dataApiInstanceBuildDuration, dataApiSchemaBuildDuration, GraphQLManager.countGraphQLSchemaLines(dataApiSchema), schemaApiInstanceBuildDuration, schemaApiSchemaBuildDuration, GraphQLManager.countGraphQLSchemaLines(schemaApiSchema));
            Assert.isPremiseValid((!this.catalogBuildStatistics.containsKey(catalogName) ? 1 : 0) != 0, () -> new GraphQLInternalError("No build statistics found for catalog `" + catalogName + "`"));
            this.catalogBuildStatistics.put(catalogName, schemaBuildStatistics);
        }
        catch (EvitaInternalError ex) {
            log.error("Catalog `" + catalogName + "` is corrupted and will not accessible by GraphQL API.", (Throwable)ex);
            this.graphQLRouter.unregisterCatalogApis(catalogName);
            this.catalogBuildStatistics.remove(catalogName);
        }
    }

    public void refreshCatalog(@Nonnull String catalogName) {
        boolean catalogRegistered = this.registeredCatalogs.contains(catalogName);
        if (!catalogRegistered) {
            log.info("Could not refresh existing catalog `{}`. Registering new one instead...", (Object)catalogName);
            this.registerCatalog(catalogName);
            return;
        }
        CatalogContract catalog = this.evita.getCatalogInstanceOrThrowException(catalogName);
        long dataApiInstanceBuildStartTime = System.currentTimeMillis();
        long dataApiSchemaBuildStartTime = System.currentTimeMillis();
        GraphQLSchema dataApiSchema = new CatalogDataApiGraphQLSchemaBuilder(this.graphQLConfig, this.evita, catalog).build();
        long dataApiSchemaBuildDuration = System.currentTimeMillis() - dataApiSchemaBuildStartTime;
        GraphQL newDataApi = new CatalogGraphQLBuilder(this.evita, catalog, dataApiSchema).build(this.graphQLConfig);
        this.graphQLRouter.refreshCatalogApi(catalogName, GraphQLInstanceType.DATA, newDataApi);
        long dataApiInstanceBuildDuration = System.currentTimeMillis() - dataApiInstanceBuildStartTime;
        long schemaApiInstanceBuildStartTime = System.currentTimeMillis();
        long schemaApiSchemaBuildStartTime = System.currentTimeMillis();
        GraphQLSchema schemaApiSchema = new CatalogSchemaApiGraphQLSchemaBuilder(this.graphQLConfig, this.evita, catalog).build();
        long schemaApiSchemaBuildDuration = System.currentTimeMillis() - schemaApiSchemaBuildStartTime;
        GraphQL newSchemaApi = new CatalogGraphQLBuilder(this.evita, catalog, schemaApiSchema).build(this.graphQLConfig);
        this.graphQLRouter.refreshCatalogApi(catalogName, GraphQLInstanceType.SCHEMA, newSchemaApi);
        long schemaApiInstanceBuildDuration = System.currentTimeMillis() - schemaApiInstanceBuildStartTime;
        CatalogBuildStatistics buildStatistics = this.catalogBuildStatistics.get(catalogName);
        Assert.isPremiseValid((buildStatistics != null ? 1 : 0) != 0, () -> new GraphQLInternalError("No build statistics found for catalog `" + catalogName + "`"));
        buildStatistics.refresh(dataApiInstanceBuildDuration, dataApiSchemaBuildDuration, GraphQLManager.countGraphQLSchemaLines(dataApiSchema), schemaApiInstanceBuildDuration, schemaApiSchemaBuildDuration, GraphQLManager.countGraphQLSchemaLines(schemaApiSchema));
    }

    public void unregisterCatalog(@Nonnull String catalogName) {
        boolean catalogRegistered = this.registeredCatalogs.remove(catalogName);
        if (catalogRegistered) {
            this.graphQLRouter.unregisterCatalogApis(catalogName);
            this.catalogBuildStatistics.remove(catalogName);
        }
    }

    public void emitObservabilityEvents() {
        Assert.isPremiseValid((this.systemBuildStatistics != null ? 1 : 0) != 0, () -> new GraphQLInternalError("No build statistics for system API found."));
        if (!this.systemBuildStatistics.reported().get()) {
            new BuiltEvent(GraphQLInstanceType.SYSTEM, BuiltEvent.BuildType.NEW, this.systemBuildStatistics.instanceBuildDuration(), this.systemBuildStatistics.schemaBuildDuration(), this.systemBuildStatistics.schemaDslLines()).commit();
            this.systemBuildStatistics.markAsReported();
        }
        this.catalogBuildStatistics.keySet().forEach(this::emitObservabilityEvents);
    }

    public void emitObservabilityEvents(@Nonnull String catalogName) {
        CatalogBuildStatistics buildStatistics = this.catalogBuildStatistics.get(catalogName);
        Assert.isPremiseValid((buildStatistics != null ? 1 : 0) != 0, () -> new GraphQLInternalError("No build statistics found for catalog `" + catalogName + "`"));
        BuiltEvent.BuildType buildType = buildStatistics.buildCount().get() == 1 ? BuiltEvent.BuildType.NEW : BuiltEvent.BuildType.REFRESH;
        new BuiltEvent(catalogName, GraphQLInstanceType.DATA, buildType, buildStatistics.dataApiSchemaBuildDuration().get(), buildStatistics.dataApiSchemaBuildDuration().get(), buildStatistics.dataApiSchemaDslLines().get()).commit();
        new BuiltEvent(catalogName, GraphQLInstanceType.SCHEMA, buildType, buildStatistics.schemaApiSchemaBuildDuration().get(), buildStatistics.schemaApiSchemaBuildDuration().get(), buildStatistics.schemaApiSchemaDslLines().get()).commit();
    }

    private static long countGraphQLSchemaLines(@Nonnull GraphQLSchema schema) {
        return GraphQLSchemaPrinter.print(schema).lines().count();
    }

    private record SystemBuildStatistics(@Nonnull AtomicBoolean reported, long instanceBuildDuration, long schemaBuildDuration, long schemaDslLines) {
        public static SystemBuildStatistics createNew(long instanceBuildDuration, long schemaBuildDuration, long schemaDslLines) {
            return new SystemBuildStatistics(new AtomicBoolean(false), instanceBuildDuration, schemaBuildDuration, schemaDslLines);
        }

        public void markAsReported() {
            this.reported.set(true);
        }
    }

    private record CatalogBuildStatistics(@Nonnull AtomicInteger buildCount, @Nonnull AtomicLong dataApiInstanceBuildDuration, @Nonnull AtomicLong dataApiSchemaBuildDuration, @Nonnull AtomicLong dataApiSchemaDslLines, @Nonnull AtomicLong schemaApiInstanceBuildDuration, @Nonnull AtomicLong schemaApiSchemaBuildDuration, @Nonnull AtomicLong schemaApiSchemaDslLines) {
        public static CatalogBuildStatistics createNew(long dataApiInstanceBuildDuration, long dataApiSchemaBuildDuration, long dataApiSchemaDslLines, long schemaApiInstanceBuildDuration, long schemaApiSchemaBuildDuration, long schemaApiSchemaDslLines) {
            return new CatalogBuildStatistics(new AtomicInteger(1), new AtomicLong(dataApiInstanceBuildDuration), new AtomicLong(dataApiSchemaBuildDuration), new AtomicLong(dataApiSchemaDslLines), new AtomicLong(schemaApiInstanceBuildDuration), new AtomicLong(schemaApiSchemaBuildDuration), new AtomicLong(schemaApiSchemaDslLines));
        }

        public void refresh(long dataApiInstanceBuildDuration, long dataApiSchemaBuildDuration, long dataApiSchemaDslLines, long schemaApiInstanceBuildDuration, long schemaApiSchemaBuildDuration, long schemaApiSchemaDslLines) {
            this.buildCount.incrementAndGet();
            this.dataApiInstanceBuildDuration.set(dataApiInstanceBuildDuration);
            this.dataApiSchemaBuildDuration.set(dataApiSchemaBuildDuration);
            this.dataApiSchemaDslLines.set(dataApiSchemaDslLines);
            this.schemaApiInstanceBuildDuration.set(schemaApiInstanceBuildDuration);
            this.schemaApiSchemaBuildDuration.set(schemaApiSchemaBuildDuration);
            this.schemaApiSchemaDslLines.set(schemaApiSchemaDslLines);
        }
    }
}

