/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.server.core.authz;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.naming.directory.SearchControls;
import org.apache.directory.server.core.CoreSession;
import org.apache.directory.server.core.filtering.EntryFilteringCursor;
import org.apache.directory.server.core.interceptor.context.SearchOperationContext;
import org.apache.directory.server.core.partition.PartitionNexus;
import org.apache.directory.server.i18n.I18n;
import org.apache.directory.shared.ldap.entry.EntryAttribute;
import org.apache.directory.shared.ldap.entry.Modification;
import org.apache.directory.shared.ldap.entry.ModificationOperation;
import org.apache.directory.shared.ldap.entry.ServerEntry;
import org.apache.directory.shared.ldap.entry.StringValue;
import org.apache.directory.shared.ldap.entry.Value;
import org.apache.directory.shared.ldap.exception.LdapException;
import org.apache.directory.shared.ldap.filter.EqualityNode;
import org.apache.directory.shared.ldap.filter.OrNode;
import org.apache.directory.shared.ldap.message.AliasDerefMode;
import org.apache.directory.shared.ldap.name.DN;
import org.apache.directory.shared.ldap.schema.AttributeType;
import org.apache.directory.shared.ldap.schema.SchemaManager;
import org.apache.directory.shared.ldap.schema.normalizers.OidNormalizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class GroupCache {
    private static final Logger LOG = LoggerFactory.getLogger(GroupCache.class);
    private static final boolean IS_DEBUG = LOG.isDebugEnabled();
    private final Map<String, Set<String>> groups = new HashMap<String, Set<String>>();
    private final PartitionNexus nexus;
    private AttributeType memberAT;
    private AttributeType uniqueMemberAT;
    private Map<String, OidNormalizer> normalizerMap;
    private DN administratorsGroupDn;
    private static final Set<DN> EMPTY_GROUPS = new HashSet<DN>();

    public GroupCache(CoreSession session) throws Exception {
        SchemaManager schemaManager = session.getDirectoryService().getSchemaManager();
        this.normalizerMap = schemaManager.getNormalizerMapping();
        this.nexus = session.getDirectoryService().getPartitionNexus();
        this.memberAT = schemaManager.lookupAttributeTypeRegistry("2.5.4.31");
        this.uniqueMemberAT = schemaManager.lookupAttributeTypeRegistry("2.5.4.50");
        this.administratorsGroupDn = this.parseNormalized("cn=Administrators,ou=groups,ou=system");
        this.initialize(session);
    }

    private DN parseNormalized(String name) throws LdapException {
        DN dn = new DN(name);
        dn.normalize(this.normalizerMap);
        return dn;
    }

    private void initialize(CoreSession session) throws Exception {
        OrNode filter = new OrNode();
        filter.addNode(new EqualityNode<String>("objectClass", new StringValue("groupOfNames")));
        filter.addNode(new EqualityNode<String>("objectClass", new StringValue("groupOfUniqueNames")));
        Set<String> suffixes = this.nexus.listSuffixes(null);
        for (String suffix : suffixes) {
            DN baseDn = new DN(suffix).normalize(this.normalizerMap);
            SearchControls ctls = new SearchControls();
            ctls.setSearchScope(2);
            SearchOperationContext searchOperationContext = new SearchOperationContext(session, baseDn, filter, ctls);
            searchOperationContext.setAliasDerefMode(AliasDerefMode.DEREF_ALWAYS);
            EntryFilteringCursor results = this.nexus.search(searchOperationContext);
            while (results.next()) {
                ServerEntry result = (ServerEntry)results.get();
                DN groupDn = result.getDn().normalize(this.normalizerMap);
                EntryAttribute members = this.getMemberAttribute(result);
                if (members != null) {
                    HashSet<String> memberSet = new HashSet<String>(members.size());
                    this.addMembers(memberSet, members);
                    this.groups.put(groupDn.getNormName(), memberSet);
                    continue;
                }
                LOG.warn("Found group '{}' without any member or uniqueMember attributes", (Object)groupDn.getName());
            }
            results.close();
        }
        if (IS_DEBUG) {
            LOG.debug("group cache contents on startup:\n {}", this.groups);
        }
    }

    private EntryAttribute getMemberAttribute(ServerEntry entry) throws LdapException {
        EntryAttribute oc = entry.get("objectClass");
        if (oc == null) {
            EntryAttribute member = entry.get(this.memberAT);
            if (member != null) {
                return member;
            }
            EntryAttribute uniqueMember = entry.get(this.uniqueMemberAT);
            if (uniqueMember != null) {
                return uniqueMember;
            }
            return null;
        }
        if (oc.contains("groupOfNames") || oc.contains("2.5.6.9")) {
            return entry.get(this.memberAT);
        }
        if (oc.contains("groupOfUniqueNames") || oc.contains("2.5.6.17")) {
            return entry.get(this.uniqueMemberAT);
        }
        return null;
    }

    private void addMembers(Set<String> memberSet, EntryAttribute members) throws LdapException {
        for (Value value : members) {
            String memberDn = value.getString();
            try {
                memberDn = this.parseNormalized(memberDn).getNormName();
            }
            catch (LdapException e) {
                LOG.warn("Malformed member DN in groupOf[Unique]Names entry.  Member not added to GroupCache.", (Throwable)e);
            }
            memberSet.add(memberDn);
        }
    }

    private void removeMembers(Set<String> memberSet, EntryAttribute members) throws LdapException {
        for (Value value : members) {
            String memberDn = value.getString();
            try {
                memberDn = this.parseNormalized(memberDn).getNormName();
            }
            catch (LdapException e) {
                LOG.warn("Malformed member DN in groupOf[Unique]Names entry.  Member not removed from GroupCache.", (Throwable)e);
            }
            memberSet.remove(memberDn);
        }
    }

    public void groupAdded(DN name, ServerEntry entry) throws LdapException {
        EntryAttribute members = this.getMemberAttribute(entry);
        if (members == null) {
            return;
        }
        HashSet<String> memberSet = new HashSet<String>(members.size());
        this.addMembers(memberSet, members);
        this.groups.put(name.getNormName(), memberSet);
        if (IS_DEBUG) {
            LOG.debug("group cache contents after adding '{}' :\n {}", (Object)name.getName(), this.groups);
        }
    }

    public void groupDeleted(DN name, ServerEntry entry) throws LdapException {
        EntryAttribute members = this.getMemberAttribute(entry);
        if (members == null) {
            return;
        }
        this.groups.remove(name.getNormName());
        if (IS_DEBUG) {
            LOG.debug("group cache contents after deleting '{}' :\n {}", (Object)name.getName(), this.groups);
        }
    }

    private void modify(Set<String> memberSet, ModificationOperation modOp, EntryAttribute members) throws LdapException {
        switch (modOp) {
            case ADD_ATTRIBUTE: {
                this.addMembers(memberSet, members);
                break;
            }
            case REPLACE_ATTRIBUTE: {
                if (members.size() <= 0) break;
                memberSet.clear();
                this.addMembers(memberSet, members);
                break;
            }
            case REMOVE_ATTRIBUTE: {
                this.removeMembers(memberSet, members);
                break;
            }
            default: {
                throw new InternalError(I18n.err(I18n.ERR_235, new Object[]{modOp}));
            }
        }
    }

    public void groupModified(DN name, List<Modification> mods, ServerEntry entry, SchemaManager schemaManager) throws LdapException {
        EntryAttribute members = null;
        String memberAttrId = null;
        EntryAttribute oc = entry.get("objectClass");
        if (oc.contains("groupOfNames")) {
            members = entry.get(this.memberAT);
            memberAttrId = "member";
        }
        if (oc.contains("groupOfUniqueNames")) {
            members = entry.get(this.uniqueMemberAT);
            memberAttrId = "uniqueMember";
        }
        if (members == null) {
            return;
        }
        for (Modification modification : mods) {
            if (!memberAttrId.equalsIgnoreCase(modification.getAttribute().getId())) continue;
            Set<String> memberSet = this.groups.get(name.getNormName());
            if (memberSet == null) break;
            this.modify(memberSet, modification.getOperation(), modification.getAttribute());
            break;
        }
        if (IS_DEBUG) {
            LOG.debug("group cache contents after modifying '{}' :\n {}", (Object)name.getName(), this.groups);
        }
    }

    public void groupModified(DN name, ModificationOperation modOp, ServerEntry mods) throws LdapException {
        EntryAttribute members = this.getMemberAttribute(mods);
        if (members == null) {
            return;
        }
        Set<String> memberSet = this.groups.get(name.getNormName());
        if (memberSet != null) {
            this.modify(memberSet, modOp, members);
        }
        if (IS_DEBUG) {
            LOG.debug("group cache contents after modifying '{}' :\n {}", (Object)name.getName(), this.groups);
        }
    }

    public final boolean isPrincipalAnAdministrator(DN principalDn) {
        if (principalDn.getNormName().equals("0.9.2342.19200300.100.1.1=admin,2.5.4.11=system")) {
            return true;
        }
        Set<String> members = this.groups.get(this.administratorsGroupDn.getNormName());
        if (members == null) {
            LOG.warn("What do you mean there is no administrators group? This is bad news.");
            return false;
        }
        return members.contains(principalDn.getNormName());
    }

    public Set<DN> getGroups(String member) throws LdapException {
        DN normMember;
        try {
            normMember = this.parseNormalized(member);
        }
        catch (LdapException e) {
            LOG.warn("Malformed member DN.  Could not find groups for member '{}' in GroupCache. Returning empty set for groups!", (Object)member, (Object)e);
            return EMPTY_GROUPS;
        }
        HashSet<DN> memberGroups = null;
        for (String group : this.groups.keySet()) {
            Set<String> members = this.groups.get(group);
            if (members == null || !members.contains(normMember.getNormName())) continue;
            if (memberGroups == null) {
                memberGroups = new HashSet<DN>();
            }
            memberGroups.add(this.parseNormalized(group));
        }
        if (memberGroups == null) {
            return EMPTY_GROUPS;
        }
        return memberGroups;
    }

    public boolean groupRenamed(DN oldName, DN newName) {
        Set<String> members = this.groups.remove(oldName.getNormName());
        if (members != null) {
            this.groups.put(newName.getNormName(), members);
            if (IS_DEBUG) {
                LOG.debug("group cache contents after renaming '{}' :\n{}", (Object)oldName.getName(), this.groups);
            }
            return true;
        }
        return false;
    }
}

