/*
 * Copyright 2017-2020 FIX Protocol Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 *
 */
package io.fixprotocol.orchestra.message;

import java.util.List;
import io.fixprotocol._2020.orchestra.repository.CodeSetType;
import io.fixprotocol._2020.orchestra.repository.CodeType;
import io.fixprotocol.orchestra.model.FixNode;
import io.fixprotocol.orchestra.model.FixType;
import io.fixprotocol.orchestra.model.FixValue;
import io.fixprotocol.orchestra.model.FixValueFactory;
import io.fixprotocol.orchestra.model.ModelException;
import io.fixprotocol.orchestra.model.PathStep;
import io.fixprotocol.orchestra.model.Scope;

/**
 * Scope for a code set
 *
 * @author Don Mendelson
 *
 */
public class CodeSetScope implements Scope {

  private final CodeSetType codeSet;
  private Scope parent;

  /**
   * Constructor
   *
   * @param codeSet from metadata
   */
  public CodeSetScope(CodeSetType codeSet) {
    this.codeSet = codeSet;
  }

  /*
   * (non-Javadoc)
   *
   * @see
   * io.fixprotocol.orchestra.dsl.antlr.Scope#assign(io.fixprotocol.orchestra.dsl.antlr.PathStep,
   * io.fixprotocol.orchestra.dsl.antlr.FixValue)
   */
  @Override
  public FixValue<?> assign(PathStep arg0, FixValue<?> arg1) throws ModelException {
    throw new UnsupportedOperationException("Message structure is immutable");
  }

  /*
   * (non-Javadoc)
   *
   * @see java.lang.AutoCloseable#close()
   */
  @Override
  public void close() throws Exception {
    if (parent != null) {
      parent.remove(new PathStep(codeSet.getName()));
    }
  }

  /*
   * (non-Javadoc)
   *
   * @see io.fixprotocol.orchestra.dsl.antlr.FixNode#getName()
   */
  @Override
  public String getName() {
    return codeSet.getName();
  }

  /*
   * (non-Javadoc)
   *
   * @see io.fixprotocol.orchestra.dsl.antlr.Scope#nest(io.fixprotocol.orchestra.dsl.antlr.PathStep,
   * io.fixprotocol.orchestra.dsl.antlr.Scope)
   */
  @Override
  public Scope nest(PathStep arg0, Scope arg1) {
    throw new UnsupportedOperationException("Message structure is immutable");
  }

  /*
   * (non-Javadoc)
   *
   * @see
   * io.fixprotocol.orchestra.dsl.antlr.Scope#remove(io.fixprotocol.orchestra.dsl.antlr.PathStep)
   */
  @Override
  public FixNode remove(PathStep arg0) {
    throw new UnsupportedOperationException("Message structure is immutable");
  }

  /*
   * (non-Javadoc)
   *
   * @see
   * io.fixprotocol.orchestra.dsl.antlr.Scope#resolve(io.fixprotocol.orchestra.dsl.antlr.PathStep)
   */
  @SuppressWarnings("unchecked")
  @Override
  public FixValue<?> resolve(PathStep pathStep) {
    final String dataTypeString = codeSet.getType();
    final FixType dataType = FixType.forName(dataTypeString);
    final String name = pathStep.getName();
    final List<CodeType> codes = codeSet.getCode();
    for (final CodeType code : codes) {
      if (code.getName().equals(name)) {
        @SuppressWarnings("rawtypes")
        FixValue fixValue;
        try {
          fixValue = FixValueFactory.create(name, dataType, dataType.getValueClass());
          fixValue.setValue(dataType.getValueClass().cast(dataType.fromString(code.getValue())));
          return fixValue;
        } catch (final ModelException e) {
          return null;
        }
      }
    }
    return null;
  }

  /*
   * (non-Javadoc)
   *
   * @see
   * io.fixprotocol.orchestra.dsl.antlr.Scope#setParent(io.fixprotocol.orchestra.dsl.antlr.Scope)
   */
  @Override
  public void setParent(Scope parent) {
    this.parent = parent;
  }

}
