package ru.foodtechlab.lib.auth.integration.inner.authSessions;

import com.rcore.domain.commons.port.dto.SearchResult;
import com.rcore.domain.commons.usecase.model.FiltersInputValues;
import com.rcore.domain.commons.usecase.model.IdInputValues;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.integration.core.authorizationSession.AuthorizationSessionServiceFacade;
import ru.foodtechlab.lib.auth.integration.inner.authSessions.mapper.AuthorizationSessionResponseMapper;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.AuthSessionEntity;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.ClientInfo;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.LoginDetails;
import ru.foodtechlab.lib.auth.service.domain.auth.port.filter.AuthorizationSessionFilters;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.authSession.DeleteAuthSessionUseCase;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.authSession.FindAuthSessionsUseCase;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.authSession.ForceFindAuthSessionByIdUseCase;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.entity.ConfirmationCodeEntity;
import ru.foodtechlab.lib.auth.service.domain.credential.entity.PhoneNumber;
import ru.foodtechlab.lib.auth.service.facade.authorizationSession.dto.requests.*;
import ru.foodtechlab.lib.auth.service.facade.authorizationSession.dto.responses.AuthorizationSessionResponse;

import java.util.Optional;
import java.util.stream.Collectors;

@Component
public class InnerAuthorizationSessionServiceFacade implements AuthorizationSessionServiceFacade {

    private final AuthorizationSessionResponseMapper mapper;
    private final FindAuthSessionsUseCase findAuthSessions;
    private final ForceFindAuthSessionByIdUseCase forceFindAuthSessionByIdUseCase;
    private final DeleteAuthSessionUseCase deleteAuthSessionById;
    private final DeleteAuthSessionUseCase deleteAuthSessionByPhoneNumber;
    private final DeleteAuthSessionUseCase deleteAuthSessionByUsername;
    private final DeleteAuthSessionUseCase deleteAuthSessionByEmail;
    private final DeleteAuthSessionUseCase deleteAuthSessionByIp;
    private final DeleteAuthSessionUseCase deleteAuthSessionByDeviceId;

    public InnerAuthorizationSessionServiceFacade(
            AuthorizationSessionResponseMapper mapper,
            FindAuthSessionsUseCase findAuthSessions,
            ForceFindAuthSessionByIdUseCase forceFindAuthSessionByIdUseCase,
            @Qualifier("deleteAuthSessionByIdUseCase") DeleteAuthSessionUseCase deleteAuthSessionById,
            @Qualifier("deleteAuthSessionByPhoneNumberUseCase") DeleteAuthSessionUseCase deleteAuthSessionByPhoneNumber,
            @Qualifier("deleteAuthSessionByUsernameUseCase") DeleteAuthSessionUseCase deleteAuthSessionByUsername,
            @Qualifier("deleteAuthSessionByEmailUseCase") DeleteAuthSessionUseCase deleteAuthSessionByEmail,
            @Qualifier("deleteAuthSessionByIpUseCase") DeleteAuthSessionUseCase deleteAuthSessionByIp,
            @Qualifier("deleteAuthSessionByDeviceIdUseCase") DeleteAuthSessionUseCase deleteAuthSessionByDeviceId
    ) {
        this.mapper = mapper;
        this.findAuthSessions = findAuthSessions;
        this.forceFindAuthSessionByIdUseCase = forceFindAuthSessionByIdUseCase;
        this.deleteAuthSessionById = deleteAuthSessionById;
        this.deleteAuthSessionByPhoneNumber = deleteAuthSessionByPhoneNumber;
        this.deleteAuthSessionByUsername = deleteAuthSessionByUsername;
        this.deleteAuthSessionByEmail = deleteAuthSessionByEmail;
        this.deleteAuthSessionByIp = deleteAuthSessionByIp;
        this.deleteAuthSessionByDeviceId = deleteAuthSessionByDeviceId;
    }

    @Override
    public SearchResult<AuthorizationSessionResponse> find(SearchAuthorizationSessionsWithFiltersRequest searchFilters) {

        var result = findAuthSessions.execute(
                FiltersInputValues.of(AuthorizationSessionFilters.builder()
                        .offset(searchFilters.getOffset())
                        .query(searchFilters.getQuery())
                        .sortDirection(searchFilters.getSortDirection())
                        .sortName(searchFilters.getSortName())
                        .limit(searchFilters.getLimit())
                        .credentialId(searchFilters.getCredentialId())
                        .confirmationCodeType(Optional.ofNullable(searchFilters.getConfirmationCodeType())
                                .map(c -> ConfirmationCodeEntity.Type.valueOf(c.name()))
                                .orElse(null))
                        .clientInfo(ClientInfo.builder()
                                .ip(searchFilters.getIpV4())
                                .deviceId(searchFilters.getDeviceId())
                                .platform(Optional.ofNullable(searchFilters.getApplicationDetails())
                                        .map(AuthorizationSessionResponse.ApplicationDetails::getPlatform)
                                        .orElse(null))
                                .versionName(Optional.ofNullable(searchFilters.getApplicationDetails())
                                        .map(AuthorizationSessionResponse.ApplicationDetails::getVersionName)
                                        .orElse(null))
                                .application(Optional.ofNullable(searchFilters.getApplicationDetails())
                                        .map(AuthorizationSessionResponse.ApplicationDetails::getName)
                                        .orElse(null))
                                .build())
                        .dateFrom(searchFilters.getDateFrom())
                        .dateTo(searchFilters.getDateTo())
                        .deleted(searchFilters.getIsDeleted())
                        .loginType(Optional.ofNullable(searchFilters.getLoginType())
                                .map(l -> AuthSessionEntity.LoginType.valueOf(l.name()))
                                .orElse(null))
                        .status(Optional.ofNullable(searchFilters.getStatus())
                                .map(s -> AuthSessionEntity.Status.valueOf(s.name()))
                                .orElse(null))
                        .type(Optional.ofNullable(searchFilters.getType())
                                .map(t -> AuthSessionEntity.Type.valueOf(t.name()))
                                .orElse(null))
                        .isRegistrationAllowed(searchFilters.getIsRegistrationAllowed())
                        .deleted(searchFilters.getIsDeleted())
                        .roleCode(searchFilters.getRoleCode())
                        .build()));
        return SearchResult.withItemsAndCount(
                result.getResult().getItems().stream().map(mapper::map).collect(Collectors.toList()),
                result.getResult().getCount()
        );
    }

    @Override
    public Optional<AuthorizationSessionResponse> findById(String id) {
        var result = forceFindAuthSessionByIdUseCase.execute(IdInputValues.of(id));
        return result.getEntity().map(mapper::map);
    }

    @Override
    public void disable(String id) {
        deleteAuthSessionById.execute(DeleteAuthSessionUseCase.InputValues.id(id));
    }

    @Override
    public void disableByPhone(DisableByPhoneRequest command) {
        deleteAuthSessionByPhoneNumber.execute(
                DeleteAuthSessionUseCase.InputValues.loginDetails(
                        LoginDetails.builder()
                                .phoneNumber(new PhoneNumber(command.getPhoneNumber(), null))
                                .build()
                ));
    }

    @Override
    public void disableByUsername(DisableByUsernameRequest command) {
        deleteAuthSessionByUsername.execute(
                DeleteAuthSessionUseCase.InputValues.loginDetails(
                        LoginDetails.builder()
                                .username(command.getUsername())
                                .build()
                ));
    }

    @Override
    public void disableByEmail(DisableByEmailRequest command) {
        deleteAuthSessionByEmail.execute(
                DeleteAuthSessionUseCase.InputValues.loginDetails(LoginDetails.builder()
                        .email(command.getEmail())
                        .build()
                ));
    }

    @Override
    public void disableByIp(DisableByIpRequest command) {
        deleteAuthSessionByIp.execute(
                DeleteAuthSessionUseCase.InputValues.clientInfo(ClientInfo.builder()
                        .ip(command.getIpV4())
                        .build()
                ));
    }

    @Override
    public void disableByDeviceId(DisableByDeviceIdRequest command) {
        deleteAuthSessionByDeviceId.execute(
                DeleteAuthSessionUseCase.InputValues.clientInfo(ClientInfo.builder()
                        .deviceId(command.getDeviceId())
                        .build()
                ));
    }
}
