package ru.foodtechlab.lib.auth.integration.inner.authorization;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.integration.core.authorizartion.AuthorizationServiceFacade;
import ru.foodtechlab.lib.auth.integration.core.authorizartion.exception.AuthServiceException;
import ru.foodtechlab.lib.auth.integration.core.authorizartion.exception.BadCredentialsExceptions;
import ru.foodtechlab.lib.auth.integration.inner.authorization.mapper.BasicAuthorizationResponseMapper;
import ru.foodtechlab.lib.auth.integration.inner.authorization.mapper.ClientInfoMapper;
import ru.foodtechlab.lib.auth.integration.inner.authorization.mapper.ConfirmationCodeTimerResponseMapper;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.AuthSessionEntity;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.LoginDetails;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.*;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.requests.*;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.responses.BasicAuthorizationResponse;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.responses.ConfirmationCodeTimerResponse;

@Component
public class InnerAuthorizationServiceFacade implements AuthorizationServiceFacade {

    private final BasicAuthorizationResponseMapper basicAuthorizationResponseMapper;
    private final ConfirmationCodeTimerResponseMapper confirmationCodeTimerResponseMapper;
    private final PasswordAuthUseCase phoneNumberPasswordAuthUseCase;
    private final InitSignUpUseCase initSignUpByPhoneNumber;
    private final InitSignUpUseCase initSignUpByEmail;
    private final SignUpConfirmationUseCase signUpConfirmationUseCase;
    private final PasswordAuthUseCase usernamePasswordAuthorizationUseCase;
    private final PasswordAuthUseCase emailPasswordAuthorizationUseCase;
    private final RefreshAccessTokenUseCase refreshAccessTokenUseCase;
    private final LogoutUseCase logoutUseCase;

    public InnerAuthorizationServiceFacade(
            BasicAuthorizationResponseMapper basicAuthorizationResponseMapper,
            ConfirmationCodeTimerResponseMapper confirmationCodeTimerResponseMapper,
            @Qualifier("phoneNumberPasswordAuthorizationUseCase") PasswordAuthUseCase phoneNumberPasswordAuthUseCase,
            @Qualifier("initSignUpByPhoneNumberUseCase") InitSignUpUseCase initSignUpByPhoneNumber,
            @Qualifier("initSignUpByEmailUseCase") InitSignUpUseCase initSignUpByEmail,
            SignUpConfirmationUseCase signUpConfirmationUseCase,
            @Qualifier("usernamePasswordAuthorizationUseCase") PasswordAuthUseCase usernamePasswordAuthorizationUseCase,
            @Qualifier("emailPasswordAuthorizationUseCase") PasswordAuthUseCase emailPasswordAuthorizationUseCase,
            RefreshAccessTokenUseCase refreshAccessTokenUseCase,
            LogoutUseCase logoutUseCase
    ) {
        this.basicAuthorizationResponseMapper = basicAuthorizationResponseMapper;
        this.confirmationCodeTimerResponseMapper = confirmationCodeTimerResponseMapper;
        this.phoneNumberPasswordAuthUseCase = phoneNumberPasswordAuthUseCase;
        this.initSignUpByPhoneNumber = initSignUpByPhoneNumber;
        this.initSignUpByEmail = initSignUpByEmail;
        this.signUpConfirmationUseCase = signUpConfirmationUseCase;
        this.usernamePasswordAuthorizationUseCase = usernamePasswordAuthorizationUseCase;
        this.emailPasswordAuthorizationUseCase = emailPasswordAuthorizationUseCase;
        this.refreshAccessTokenUseCase = refreshAccessTokenUseCase;
        this.logoutUseCase = logoutUseCase;
    }

    @Override
    public BasicAuthorizationResponse phoneNumberAuthorization(PhoneNumberAuthorizationRequest command) throws AuthServiceException, BadCredentialsExceptions {
        var result = phoneNumberPasswordAuthUseCase.execute(PasswordAuthUseCase.InputValues.builder()
                .loginType(AuthSessionEntity.LoginType.PHONE_NUMBER)
                .loginDetails(LoginDetails.phone(command.getPhoneNumber(), command.getIsoTwoLetterCountryCode()))
                .clientInfo(ClientInfoMapper.map(command.getDeviceId(), command.getIp(), command.getApplicationDetails()))
                .password(command.getPassword())
                .isRegistrationAllowed(command.getIsRegistrationAllowed())
                .targetRoleCode(command.getRoleCode())
                .build());
        return basicAuthorizationResponseMapper.map(result);
    }

    @Override
    public ConfirmationCodeTimerResponse singUpByPhoneNumber(SignUpByPhoneNumberRequest command) {
        var result = initSignUpByPhoneNumber.execute(
                InitSignUpUseCase.InputValues.builder()
                        .loginType(AuthSessionEntity.LoginType.PHONE_NUMBER)
                        .loginDetails(LoginDetails.phone(command.getPhoneNumber(), command.getIsoTwoLetterCountryCode()))
                        .targetRoleCode(command.getRoleCode())
                        .isRegistrationAllowed(command.getIsRegistrationAllowed())
                        .clientInfo(ClientInfoMapper.map(command.getDeviceId(), command.getIp(), command.getApplicationDetails()))
                        .build());
        return confirmationCodeTimerResponseMapper.map(result);
    }

    @Override
    public ConfirmationCodeTimerResponse singUpByEmail(SignUpByEmailRequest command) {
        var result = initSignUpByEmail.execute(
                InitSignUpUseCase.InputValues.builder()
                        .loginType(AuthSessionEntity.LoginType.EMAIL)
                        .loginDetails(LoginDetails.email(command.getEmail()))
                        .targetRoleCode(command.getRoleCode())
                        .isRegistrationAllowed(command.getIsRegistrationAllowed())
                        .clientInfo(ClientInfoMapper.map(command.getDeviceId(), command.getIp(), command.getApplicationDetails()))
                        .build());
        return confirmationCodeTimerResponseMapper.map(result);
    }

    @Override
    public BasicAuthorizationResponse signUpConfirmationByPhoneNumber(SignUpConfirmationByPhoneNumberRequest command) {
        var result = signUpConfirmationUseCase.execute(
                SignUpConfirmationUseCase.InputValues.builder()
                        .loginType(AuthSessionEntity.LoginType.PHONE_NUMBER)
                        .loginDetails(LoginDetails.phone(command.getPhoneNumber(), command.getIsoTwoLetterCountryCode()))
                        .clientInfo(ClientInfoMapper.map(command.getDeviceId()))
                        .confirmationCode(command.getConfirmationCode())
                        .build());
        return basicAuthorizationResponseMapper.map(result);
    }

    @Override
    public BasicAuthorizationResponse signUpConfirmationByEmail(SignUpConfirmationByEmailRequest command) {
        var result = signUpConfirmationUseCase.execute(
                SignUpConfirmationUseCase.InputValues.builder()
                        .loginType(AuthSessionEntity.LoginType.EMAIL)
                        .loginDetails(LoginDetails.email(command.getEmail()))
                        .clientInfo(ClientInfoMapper.map(command.getDeviceId()))
                        .confirmationCode(command.getConfirmationCode())
                        .build());
        return basicAuthorizationResponseMapper.map(result);
    }

    @Override
    public BasicAuthorizationResponse usernameAuthorization(UsernameAuthorizationRequest command) throws AuthServiceException, BadCredentialsExceptions {
        var result = usernamePasswordAuthorizationUseCase.execute(
                PasswordAuthUseCase.InputValues.builder()
                        .loginType(AuthSessionEntity.LoginType.USERNAME)
                        .loginDetails(LoginDetails.username(command.getUsername()))
                        .clientInfo(ClientInfoMapper.map(command.getDeviceId(), command.getIp(), command.getApplicationDetails()))
                        .password(command.getPassword())
                        .targetRoleCode(command.getRoleCode())
                        .isRegistrationAllowed(command.getIsRegistrationAllowed())
                        .build());
        return basicAuthorizationResponseMapper.map(result);
    }

    @Override
    public BasicAuthorizationResponse emailAuthorization(EmailAuthorizationRequest command) throws AuthServiceException, BadCredentialsExceptions {
        var result = emailPasswordAuthorizationUseCase.execute(
                PasswordAuthUseCase.InputValues.builder()
                        .loginType(AuthSessionEntity.LoginType.EMAIL)
                        .loginDetails(LoginDetails.email(command.getEmail()))
                        .clientInfo(ClientInfoMapper.map(command.getDeviceId(), command.getIp(), command.getApplicationDetails()))
                        .password(command.getPassword())
                        .targetRoleCode(command.getRoleCode())
                        .isRegistrationAllowed(command.getIsRegistrationAllowed())
                        .build());
        return basicAuthorizationResponseMapper.map(result);
    }

    @Override
    public BasicAuthorizationResponse refreshToken(RefreshTokenRequest command) throws AuthServiceException {
        var result = refreshAccessTokenUseCase.execute(
                RefreshAccessTokenUseCase.InputValues.of(command.getRefreshToken()));
        return basicAuthorizationResponseMapper.map(result);
    }

    @Override
    public void logout(LogoutRequest command) throws AuthServiceException {
        logoutUseCase.execute(LogoutUseCase.InputValues.of(command.getAccessToken()));
    }
}

