package ru.foodtechlab.lib.auth.integration.inner.config;

import com.rcore.commons.utils.EmailValidator;
import com.rcore.commons.utils.PasswordCryptographer;
import com.rcore.commons.utils.PhoneNumberValidator;
import lombok.RequiredArgsConstructor;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import ru.foodtechlab.lib.auth.service.domain.auth.exceptions.AuthorizationSessionNotFoundException;
import ru.foodtechlab.lib.auth.service.domain.auth.port.AuthorizationSessionRepository;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.CheckAuthLimitForLoginDetailsUseCase;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.InitSignUpUseCase;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.PasswordAuthUseCase;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.SignUpConfirmationUseCase;
import ru.foodtechlab.lib.auth.service.domain.auth.usecases.authSession.*;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.usecases.ConfirmConfirmationCodeUseCase;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.usecases.CreateConfirmationCodeUseCase;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.usecases.ExpireConfirmationCodeUseCase;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.usecases.FindCodeBySessionUseCase;
import ru.foodtechlab.lib.auth.service.domain.credential.usecases.*;
import ru.foodtechlab.lib.auth.service.domain.preference.service.DefaultConfirmationCodeTypeResolver;
import ru.foodtechlab.lib.auth.service.domain.preference.service.RoleResolver;
import ru.foodtechlab.lib.auth.service.domain.preference.usecase.GetTTLForAuthWithConfirmationUseCase;
import ru.foodtechlab.lib.auth.service.domain.token.usecases.CreateTokenPairUseCase;

import java.util.Collections;

@RequiredArgsConstructor
@Configuration
public class AuthUseCasesConfig {

    @Bean("phoneNumberPasswordAuthorizationUseCase")
    public PasswordAuthUseCase phoneNumberPasswordAuthorizationUseCase(
            CreateCredentialUseCase createCredentialUseCase,
            AddRoleToCredentialUseCase addRoleToCredentialUseCase,
            FindCredentialByPhoneNumberUseCase findCredentialByPhoneNumberUseCase,
            FindPendingOrCreateAuthSessionUseCase findPendingOrCreateAuthSessionUseCase,
            ErrorAuthSessionUseCase errorAuthSessionUseCase,
            SuccessAuthSessionUseCase successAuthSessionUseCase,
            CreateTokenPairUseCase createTokenPairUseCase,
            RoleResolver roleResolver,
            PasswordCryptographer passwordCryptographer,
            PhoneNumberValidator phoneNumberValidator
    ) {
        return new PasswordAuthUseCase(
                l -> findCredentialByPhoneNumberUseCase.execute(FindCredentialByPhoneNumberUseCase.InputValues.of(l.getPhoneNumber().getValue())).getValue(),
                findPendingOrCreateAuthSessionUseCase,
                errorAuthSessionUseCase,
                successAuthSessionUseCase,
                roleResolver,
                createTokenPairUseCase,
                passwordCryptographer,
                phoneNumberValidator,
                createCredentialUseCase,
                addRoleToCredentialUseCase
        );
    }

    @Bean("emailPasswordAuthorizationUseCase")
    public PasswordAuthUseCase emailPasswordAuthorizationUseCase(
            CreateCredentialUseCase createCredentialUseCase,
            AddRoleToCredentialUseCase addRoleToCredentialUseCase,
            FindCredentialByEmailUseCase findCredentialByEmailUseCase,
            FindPendingOrCreateAuthSessionUseCase findPendingOrCreateAuthSessionUseCase,
            ErrorAuthSessionUseCase errorAuthSessionUseCase,
            SuccessAuthSessionUseCase successAuthSessionUseCase,
            CreateTokenPairUseCase createTokenPairUseCase,
            RoleResolver roleResolver,
            PasswordCryptographer passwordCryptographer,
            PhoneNumberValidator phoneNumberValidator
    ) {
        return new PasswordAuthUseCase(
                l -> findCredentialByEmailUseCase.execute(FindCredentialByEmailUseCase.InputValues.of(l.getEmail())).getValue(),
                findPendingOrCreateAuthSessionUseCase,
                errorAuthSessionUseCase,
                successAuthSessionUseCase,
                roleResolver,
                createTokenPairUseCase,
                passwordCryptographer,
                phoneNumberValidator,
                createCredentialUseCase,
                addRoleToCredentialUseCase
        );
    }

    @Bean("usernamePasswordAuthorizationUseCase")
    public PasswordAuthUseCase usernamePasswordAuthorizationUseCase(
            CreateCredentialUseCase createCredentialUseCase,
            AddRoleToCredentialUseCase addRoleToCredentialUseCase,
            FindCredentialByUsernameUseCase findCredentialByUsernameUseCase,
            FindPendingOrCreateAuthSessionUseCase findPendingOrCreateAuthSessionUseCase,
            ErrorAuthSessionUseCase errorAuthSessionUseCase,
            SuccessAuthSessionUseCase successAuthSessionUseCase,
            CreateTokenPairUseCase createTokenPairUseCase,
            RoleResolver roleResolver,
            PasswordCryptographer passwordCryptographer,
            PhoneNumberValidator phoneNumberValidator
    ) {
        return new PasswordAuthUseCase(
                l -> findCredentialByUsernameUseCase.execute(FindCredentialByUsernameUseCase.InputValues.of(l.getUsername())).getValue(),
                findPendingOrCreateAuthSessionUseCase,
                errorAuthSessionUseCase,
                successAuthSessionUseCase,
                roleResolver,
                createTokenPairUseCase,
                passwordCryptographer,
                phoneNumberValidator,
                createCredentialUseCase,
                addRoleToCredentialUseCase
        );
    }

    @Bean
    public InitSignUpUseCase initSignUpByPhoneNumberUseCase(
            FindCredentialByPhoneNumberUseCase findCredentialByPhoneNumberUseCase,
            FindPendingOrCreateAuthSessionUseCase findPendingOrCreateAuthSessionUseCase,
            InitErrorAuthSessionUseCase initErrorAuthSessionUseCase,
            PendingConfirmationAuthSessionUseCase pendingConfirmationAuthSession,
            RoleResolver roleResolver,
            DefaultConfirmationCodeTypeResolver defaultConfirmationCodeTypeResolver,
            GetTTLForAuthWithConfirmationUseCase getTTLForAuthWithConfirmationUseCase,
            CreateConfirmationCodeUseCase createConfirmationCodeUseCase,
            CheckAuthLimitForLoginDetailsUseCase checkAuthLimitForLoginDetailsUseCase,
            PhoneNumberValidator phoneNumberValidator,
            EmailValidator emailValidator,
            FindCodeBySessionUseCase findCodeBySession
    ) {
        return new InitSignUpUseCase(
                loginDetails -> findCredentialByPhoneNumberUseCase.execute(FindCredentialByPhoneNumberUseCase.InputValues.of(loginDetails.getPhoneNumber().getValue())).getValue(),
                findPendingOrCreateAuthSessionUseCase,
                initErrorAuthSessionUseCase,
                pendingConfirmationAuthSession,
                roleResolver,
                defaultConfirmationCodeTypeResolver,
                getTTLForAuthWithConfirmationUseCase,
                createConfirmationCodeUseCase,
                checkAuthLimitForLoginDetailsUseCase,
                phoneNumberValidator,
                emailValidator,
                findCodeBySession
        );
    }

    @Bean
    public InitSignUpUseCase initSignUpByEmailUseCase(
            FindCredentialByEmailUseCase findCredentialByEmailUseCase,
            FindPendingOrCreateAuthSessionUseCase findPendingOrCreateAuthSessionUseCase,
            InitErrorAuthSessionUseCase initErrorAuthSessionUseCase,
            PendingConfirmationAuthSessionUseCase pendingConfirmationAuthSession,
            RoleResolver roleResolver,
            DefaultConfirmationCodeTypeResolver defaultConfirmationCodeTypeResolver,
            GetTTLForAuthWithConfirmationUseCase getTTLForAuthWithConfirmationUseCase,
            CreateConfirmationCodeUseCase createConfirmationCodeUseCase,
            CheckAuthLimitForLoginDetailsUseCase checkAuthLimitForLoginDetailsUseCase,
            PhoneNumberValidator phoneNumberValidator,
            EmailValidator emailValidator,
            FindCodeBySessionUseCase findCodeBySession
    ) {
        return new InitSignUpUseCase(
                loginDetails -> findCredentialByEmailUseCase.execute(FindCredentialByEmailUseCase.InputValues.of(loginDetails.getEmail())).getValue(),
                findPendingOrCreateAuthSessionUseCase,
                initErrorAuthSessionUseCase,
                pendingConfirmationAuthSession,
                roleResolver,
                defaultConfirmationCodeTypeResolver,
                getTTLForAuthWithConfirmationUseCase,
                createConfirmationCodeUseCase,
                checkAuthLimitForLoginDetailsUseCase,
                phoneNumberValidator,
                emailValidator,
                findCodeBySession
        );
    }

    @Bean
    public SignUpConfirmationUseCase signUpConfirmationUseCase(
            FindPendingAuthSessionByAuthDataUseCase findPendingAuthSessionByAuthData,
            ConfirmErrorAuthSessionUseCase confirmErrorAuthSession,
            ConfirmAuthSessionUseCase confirmAuthSession,
            ExpireAuthSessionUseCase expireAuthSession,
            ExpireConfirmationCodeUseCase expireConfirmationCode,
            ConfirmConfirmationCodeUseCase confirmConfirmationCode,
            CreateTokenPairUseCase createTokenPair,
            CreateCredentialUseCase createCredential,
            AddRoleToCredentialUseCase addRoleToCredential,
            PhoneNumberValidator phoneNumberValidator,
            EmailValidator emailValidator,
            ConfirmLoginDetailsUseCase confirmLoginDetails,
            FindCodeBySessionUseCase findCodeBySession
    ) {
        return new SignUpConfirmationUseCase(
                findPendingAuthSessionByAuthData,
                confirmErrorAuthSession,
                confirmAuthSession,
                expireAuthSession,
                expireConfirmationCode,
                confirmConfirmationCode,
                createTokenPair,
                createCredential,
                addRoleToCredential,
                phoneNumberValidator,
                emailValidator,
                confirmLoginDetails,
                findCodeBySession
        );
    }

    @Bean
    public DeleteAuthSessionUseCase deleteAuthSessionByIdUseCase(AuthorizationSessionRepository authorizationSessionRepository) {
        return new DeleteAuthSessionUseCase(i -> {
            var result = authorizationSessionRepository.findById(i.getId());
            if (result.isEmpty())
                throw new AuthorizationSessionNotFoundException();
            return Collections.singletonList(result.get());
        }, authorizationSessionRepository);
    }

    @Bean
    public DeleteAuthSessionUseCase deleteAuthSessionByPhoneNumberUseCase(AuthorizationSessionRepository authorizationSessionRepository) {
        return new DeleteAuthSessionUseCase(i -> authorizationSessionRepository.findByLoginDetailsInDay(i.getLoginDetails()), authorizationSessionRepository);
    }

    @Bean
    public DeleteAuthSessionUseCase deleteAuthSessionByEmailUseCase(AuthorizationSessionRepository authorizationSessionRepository) {
        return new DeleteAuthSessionUseCase(i -> authorizationSessionRepository.findByLoginDetailsInDay(i.getLoginDetails()), authorizationSessionRepository);
    }

    @Bean
    public DeleteAuthSessionUseCase deleteAuthSessionByUsernameUseCase(AuthorizationSessionRepository authorizationSessionRepository) {
        return new DeleteAuthSessionUseCase(i -> authorizationSessionRepository.findByLoginDetailsInDay(i.getLoginDetails()), authorizationSessionRepository);
    }

    @Bean
    public DeleteAuthSessionUseCase deleteAuthSessionByIpUseCase(AuthorizationSessionRepository authorizationSessionRepository) {
        return new DeleteAuthSessionUseCase(i -> authorizationSessionRepository.findByClientInfoInDay(i.getClientInfo()), authorizationSessionRepository);
    }

    @Bean
    public DeleteAuthSessionUseCase deleteAuthSessionByDeviceIdUseCase(AuthorizationSessionRepository authorizationSessionRepository) {
        return new DeleteAuthSessionUseCase(i -> authorizationSessionRepository.findByClientInfoInDay(i.getClientInfo()), authorizationSessionRepository);
    }

}
