package ru.foodtechlab.lib.auth.integration.inner.confirmationCode;

import com.rcore.domain.commons.port.dto.SearchResult;
import com.rcore.domain.commons.usecase.model.FiltersInputValues;
import com.rcore.domain.commons.usecase.model.IdInputValues;
import com.rcore.rest.api.commons.request.SearchApiRequest;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.integration.core.confirmationCode.ConfirmationCodeServiceFacade;
import ru.foodtechlab.lib.auth.integration.inner.confirmationCode.mapper.ConfirmationCodeMapper;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.AuthSessionEntity;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.entity.ConfirmationCodeDestinationType;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.entity.ConfirmationCodeEntity;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.port.filter.ConfirmationCodeFilters;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.usecases.FindConfirmationCodesUseCase;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.usecases.ForceFindConfirmationCodeByIdUseCase;
import ru.foodtechlab.lib.auth.service.facade.confirmationCode.dto.requests.SearchConfirmationCodeWithFiltersRequest;
import ru.foodtechlab.lib.auth.service.facade.confirmationCode.dto.responses.ConfirmationCodeResponse;

import java.util.Optional;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Component
public class InnerConfirmationCodeServiceFacade implements ConfirmationCodeServiceFacade {

    private final ConfirmationCodeMapper confirmationCodeMapper;
    private final FindConfirmationCodesUseCase findConfirmationCodesUseCase;
    private final ForceFindConfirmationCodeByIdUseCase forceFindConfirmationCodeByIdUseCase;

    @Override
    public SearchResult<ConfirmationCodeResponse> find(SearchConfirmationCodeWithFiltersRequest searchFilters) {
        var result = findConfirmationCodesUseCase.execute(
                FiltersInputValues.of(SearchApiRequest.fill(ConfirmationCodeFilters.builder(), searchFilters)
                        .authSessionId(searchFilters.getAuthSessionId())
                        .deviceId(searchFilters.getDeviceId())
                        .confirmationCodeDestinationType(Optional.ofNullable(searchFilters.getConfirmationCodeDestinationType())
                                .map(c -> ConfirmationCodeDestinationType.valueOf(c.name()))
                                .orElse(null))
                        .confirmationStatus(Optional.ofNullable(searchFilters.getConfirmationStatus())
                                .map(c -> ConfirmationCodeEntity.ConfirmationStatus.valueOf(c.name()))
                                .orElse(null))
                        .loginType(Optional.ofNullable(searchFilters.getLoginType())
                                .map(l -> AuthSessionEntity.LoginType.valueOf(l.name()))
                                .orElse(null))
                        .deleted(searchFilters.getIsDeleted())
                        .build()));
        return SearchResult.withItemsAndCount(
                result.getResult().getItems().stream().map(confirmationCodeMapper::map).collect(Collectors.toList()),
                result.getResult().getCount()
        );
    }

    @Override
    public Optional<ConfirmationCodeResponse> findById(String id) {
        var result = forceFindConfirmationCodeByIdUseCase.execute(IdInputValues.of(id));
        return result.getEntity().map(confirmationCodeMapper::map);
    }
}
