package ru.foodtechlab.lib.auth.integration.inner.credential;

import com.rcore.domain.commons.port.dto.SearchResult;
import com.rcore.domain.commons.usecase.model.FiltersInputValues;
import com.rcore.domain.commons.usecase.model.IdInputValues;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.integration.core.credential.CredentialServiceFacade;
import ru.foodtechlab.lib.auth.integration.inner.credential.mapper.CredentialResponseMapper;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.entity.ConfirmationCodeDestinationType;
import ru.foodtechlab.lib.auth.service.domain.confirmationCode.entity.ConfirmationCodeEntity;
import ru.foodtechlab.lib.auth.service.domain.credential.entity.Email;
import ru.foodtechlab.lib.auth.service.domain.credential.entity.PhoneNumber;
import ru.foodtechlab.lib.auth.service.domain.credential.port.filters.CredentialFilters;
import ru.foodtechlab.lib.auth.service.domain.credential.usecases.*;
import ru.foodtechlab.lib.auth.service.facade.credential.dto.requests.*;
import ru.foodtechlab.lib.auth.service.facade.credential.dto.responses.CredentialResponse;

import java.util.Optional;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Component
public class InnerCredentialServiceFacade implements CredentialServiceFacade {

    private final CredentialResponseMapper credentialResponseMapper;
    private final GenerateDefaultCredentialUseCase generateDefaultCredential;
    private final InitCredentialsUseCase initCredentialsUseCase;
    private final CheckAvailableInitCredentialsUseCase checkAvailableInitCredentialsUseCase;
    private final FindCredentialsUseCase findCredentialsUseCase;
    private final ForceFindCredentialByIdUseCase forceFindCredentialByIdUseCase;
    private final FindCredentialByUsernameUseCase findCredentialByUsernameUseCase;
    private final CreateCredentialUseCase createCredentialUseCase;
    private final UpdateCredentialUseCase updateCredentialUseCase;
    private final ChangeBlockStatusCredentialUseCase changeBlockStatusCredentialUseCase;
    private final ChangeCredentialPasswordUseCase changeCredentialPasswordUseCase;
    private final DeleteCredentialUseCase deleteCredentialUseCase;
    private final FindCredentialByPhoneNumberUseCase findCredentialByPhoneNumberUseCase;
    private final FindCredentialByEmailUseCase findCredentialByEmailUseCase;

    @Override
    public CredentialResponse generateDefaultCredential(String password) {
        var result = generateDefaultCredential.execute(GenerateDefaultCredentialUseCase.InputValues.of(password));
        return credentialResponseMapper.map(result.getValue());
    }

    @Override
    public CredentialResponse init(InitCredentialRequest request) {
        var result = initCredentialsUseCase.execute(
                InitCredentialsUseCase.InputValues.builder()
                        .confirmationCode(request.getConfirmationCode())
                        .email(request.getEmail())
                        .password(request.getPassword())
                        .phoneNumber(request.getPhoneNumber())
                        .isoTwoLetterCountryCode(request.getIsoTwoLetterCountryCode())
                        .username(request.getUsername())
                        .build());
        return credentialResponseMapper.map(result.getEntity());
    }

    @Override
    public Boolean checkInitAvailable() {
        return checkAvailableInitCredentialsUseCase.execute(
                CheckAvailableInitCredentialsUseCase.InputValues.builder().build())
                .getValue();
    }

    @Override
    public SearchResult<CredentialResponse> find(FindCredentialWithFiltersRequest request) {
        var result = findCredentialsUseCase.execute(
                FiltersInputValues.of(CredentialFilters.builder()
                        .query(request.getQuery())
                        .limit(request.getLimit())
                        .offset(request.getOffset())
                        .sortName(request.getSortName())
                        .sortDirection(request.getSortDirection())
                        .blocked(request.getIsBlocked())
                        .deleted(request.getIsDeleted())
                        .roleIds(request.getRoleIds())
                        .confirmationCodeDestinationType(Optional.ofNullable(request.getConfirmationCodeDestinationType()).map(c -> ConfirmationCodeDestinationType.valueOf(c.name())).orElse(null))
                        .confirmationCodeType(Optional.ofNullable(request.getConfirmationCodeType()).map(c -> ConfirmationCodeEntity.Type.valueOf(c.name())).orElse(null))
                        .build()));
        return SearchResult.withItemsAndCount(
                result.getResult().getItems().stream().map(credentialResponseMapper::map).collect(Collectors.toList()),
                result.getResult().getCount()
        );
    }

    @Override
    public Optional<CredentialResponse> findByPhoneNumber(String phoneNumber) {
        var res = findCredentialByPhoneNumberUseCase
                .execute(FindCredentialByPhoneNumberUseCase.InputValues.of(phoneNumber));
        return res.getValue().map(credentialResponseMapper::map);
    }

    @Override
    public Optional<CredentialResponse> findByEmail(String email) {
        var res = findCredentialByEmailUseCase
                .execute(FindCredentialByEmailUseCase.InputValues.of(email));
        return res.getValue().map(credentialResponseMapper::map);
    }

    @Override
    public Optional<CredentialResponse> findById(String id) {
        var result = forceFindCredentialByIdUseCase.execute(IdInputValues.of(id));
        return result.getEntity().map(credentialResponseMapper::map);
    }

    @Override
    public Optional<CredentialResponse> findByName(String name) {
        var result = findCredentialByUsernameUseCase.execute(
                FindCredentialByUsernameUseCase.InputValues.of(name));
        return result.getValue().map(credentialResponseMapper::map);
    }

    @Override
    public CredentialResponse create(CreateCredentialRequest request) {
        var result = createCredentialUseCase.execute(
                CreateCredentialUseCase.InputValues.builder()
                        .isBlocked(request.getIsBlocked())
                        .personalConfirmationCode(request.getPersonalConfirmationCode())
                        .confirmationCodeDestinationType(Optional.ofNullable(request.getConfirmationCodeDestinationType())
                                .map(c -> ConfirmationCodeDestinationType.valueOf(c.name()))
                                .orElse(null))
                        .confirmationCodeType(Optional.ofNullable(request.getConfirmationCodeType())
                                .map(c -> ConfirmationCodeEntity.Type.valueOf(c.name()))
                                .orElse(null))
                        .email(Optional.ofNullable(request.getEmail())
                                .map(e -> new Email(e.getValue(), e.getIsConfirmed()))
                                .orElse(null))
                        .password(request.getPassword())
                        .phoneNumber(Optional.ofNullable(request.getPhoneNumber())
                                .map(e -> new PhoneNumber(e.getValue(), e.getIsoTwoLetterCountryCode(), false))
                                .orElse(null))
                        .username(request.getUsername())
                        .isBlocked(request.getIsBlocked())
                        .roles(Optional.ofNullable(request.getRoles())
                                .map(r -> r.stream()
                                        .map(role -> CreateCredentialUseCase.InputValues.Role.builder()
                                                .isBlocked(role.getIsBlocked())
                                                .code(role.getCode())
                                                .roleId(role.getRoleId())
                                                .build())
                                        .collect(Collectors.toList()))
                                .orElse(null))
                        .build());
        return credentialResponseMapper.map(result.getEntity());
    }

    @Override
    public CredentialResponse update(String id, UpdateCredentialRequest request) {
        var result = updateCredentialUseCase.execute(
                UpdateCredentialUseCase.InputValues.builder()
                        .id(id)
                        .isBlocked(request.getIsBlocked())
                        .personalConfirmationCode(request.getPersonalConfirmationCode())
                        .confirmationCodeDestinationType(ConfirmationCodeDestinationType.valueOf(request.getConfirmationCodeDestinationType().name()))
                        .confirmationCodeType(ConfirmationCodeEntity.Type.valueOf(request.getConfirmationCodeType().name()))
                        .email(Optional.ofNullable(request.getEmail())
                                .map(e -> new Email(e.getValue(), e.getIsConfirmed()))
                                .orElse(null))
                        .phoneNumber(Optional.ofNullable(request.getPhoneNumber())
                                .map(e -> new PhoneNumber(e.getValue(), e.getIsoTwoLetterCountryCode(), e.getIsConfirmed()))
                                .orElse(null))
                        .username(request.getUsername())
                        .isBlocked(request.getIsBlocked())
                        .roles(Optional.ofNullable(request.getRoles())
                                .map(r -> r.stream()
                                        .map(role -> CreateCredentialUseCase.InputValues.Role.builder()
                                                .isBlocked(role.getIsBlocked())
                                                .code(role.getCode())
                                                .roleId(role.getRoleId())
                                                .build())
                                        .collect(Collectors.toList()))
                                .orElse(null))
                        .build());
        return credentialResponseMapper.map(result.getEntity());
    }

    @Override
    public CredentialResponse changeStatus(String id) {
        var result = changeBlockStatusCredentialUseCase.execute(IdInputValues.of(id));
        return credentialResponseMapper.map(result.getValue());
    }

    @Override
    public CredentialResponse changePassword(String id, ChangeCredentialPasswordRequest request) {
        var result = changeCredentialPasswordUseCase.execute(ChangeCredentialPasswordUseCase.InputValues.of(id, request.getNewPassword()));
        return credentialResponseMapper.map(result.getEntity());
    }

    @Override
    public void delete(String id) {
        deleteCredentialUseCase.execute(IdInputValues.of(id));
    }
}
