package ru.foodtechlab.lib.auth.integration.inner.role;

import com.rcore.domain.commons.port.dto.SearchResult;
import com.rcore.domain.commons.usecase.model.FiltersInputValues;
import com.rcore.domain.commons.usecase.model.IdInputValues;
import com.rcore.domain.commons.usecase.model.VoidInputValues;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.integration.core.role.RoleServiceFacade;
import ru.foodtechlab.lib.auth.integration.inner.role.mapper.RoleResponseMapper;
import ru.foodtechlab.lib.auth.service.domain.role.port.filters.RoleFilters;
import ru.foodtechlab.lib.auth.service.domain.role.usecases.*;
import ru.foodtechlab.lib.auth.service.facade.role.dto.requests.CreateRoleRequest;
import ru.foodtechlab.lib.auth.service.facade.role.dto.requests.SearchRoleWithFiltersRequest;
import ru.foodtechlab.lib.auth.service.facade.role.dto.requests.UpdateRoleRequest;
import ru.foodtechlab.lib.auth.service.facade.role.dto.responses.RoleResponse;

import java.util.Optional;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Component
public class InnerRoleServiceFacade implements RoleServiceFacade {

    private final RoleResponseMapper roleResponseMapper;
    private final GenerateDefaultRoleUseCase generateDefaultRoleUseCase;
    private final GenerateManagerRoleUseCase generateManagerRoleUseCase;
    private final FindRolesUseCase findRolesUseCase;
    private final FindRoleByIdUseCase findRoleByIdUseCase;
    private final FindRoleByCodeUseCase findRoleByCodeUseCase;
    private final CreateRoleUseCase createRoleUseCase;
    private final UpdateRoleUseCase updateRoleUseCase;
    private final DeleteRoleUseCase deleteRoleUseCase;

    @Override
    public RoleResponse generateDefaultRole() {
        var result = generateDefaultRoleUseCase.execute(new VoidInputValues());
        return roleResponseMapper.map(result.getValue());
    }

    @Override
    public RoleResponse generateManagerRole() {
        return roleResponseMapper.map(generateManagerRoleUseCase.execute(new VoidInputValues()).getValue());
    }

    @Override
    public SearchResult<RoleResponse> find(SearchRoleWithFiltersRequest request) {
        var result = findRolesUseCase.execute(
                FiltersInputValues.of(RoleFilters.builder()
                        .sortName(request.getSortName())
                        .sortDirection(request.getSortDirection())
                        .query(request.getQuery())
                        .offset(request.getOffset())
                        .limit(request.getLimit())
                        .accessIds(request.getAccessIds())
                        .isRegistrationAllowed(request.getIsRegistrationAllowed())
                        .deleted(request.getIsDeleted())
                        .build()));
        return SearchResult.withItemsAndCount(
                result.getResult().getItems().stream().map(roleResponseMapper::map).collect(Collectors.toList()),
                result.getResult().getCount()
        );
    }

    @Override
    public Optional<RoleResponse> findById(String id) {
        var result = findRoleByIdUseCase.execute(IdInputValues.of(id));
        return result.getEntity().map(roleResponseMapper::map);
    }

    @Override
    public Optional<RoleResponse> findByCode(String name) {
        var result = findRoleByCodeUseCase.execute(FindRoleByCodeUseCase.InputValues.of(name));
        return result.getEntity().map(roleResponseMapper::map);
    }

    @Override
    public RoleResponse create(CreateRoleRequest request) {
        var result = createRoleUseCase.execute(
                CreateRoleUseCase.InputValues.builder()
                        .code(request.getCode())
                        .name(request.getName())
                        .accessIds(request.getAccessIds())
                        .isRegistrationAllowed(Optional.ofNullable(request.getIsRegistrationAllowed()).orElse(false))
                        .build());
        return roleResponseMapper.map(result.getEntity());
    }

    @Override
    public RoleResponse update(String id, UpdateRoleRequest request) {
        var result = updateRoleUseCase.execute(
                UpdateRoleUseCase.InputValues.builder()
                        .id(id)
                        .code(request.getCode())
                        .name(request.getName())
                        .accessIds(request.getAccessIds())
                        .isRegistrationAllowed(Optional.ofNullable(request.getIsRegistrationAllowed()).orElse(false))
                        .build());
        return roleResponseMapper.map(result.getEntity());
    }

    @Override
    public void delete(String id) {
        deleteRoleUseCase.execute(IdInputValues.of(id));
    }
}
