package ru.foodtechlab.lib.auth.integration.inner.roleAccess;

import com.rcore.domain.commons.port.dto.SearchResult;
import com.rcore.domain.commons.usecase.model.FiltersInputValues;
import com.rcore.domain.commons.usecase.model.IdInputValues;
import com.rcore.domain.commons.usecase.model.VoidInputValues;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.integration.core.roleAccess.RoleAccessServiceFacade;
import ru.foodtechlab.lib.auth.integration.inner.roleAccess.mapper.RoleAccessResponseMapper;
import ru.foodtechlab.lib.auth.service.domain.roleAccess.entity.RoleAccessEntity;
import ru.foodtechlab.lib.auth.service.domain.roleAccess.port.filters.RoleAccessFilters;
import ru.foodtechlab.lib.auth.service.domain.roleAccess.usecase.*;
import ru.foodtechlab.lib.auth.service.facade.roleAccess.dto.requests.CreateRoleAccessRequest;
import ru.foodtechlab.lib.auth.service.facade.roleAccess.dto.requests.FindRoleAccessWithFiltersRequest;
import ru.foodtechlab.lib.auth.service.facade.roleAccess.dto.requests.UpdateRoleAccessRequest;
import ru.foodtechlab.lib.auth.service.facade.roleAccess.dto.responses.RoleAccessResponse;

import java.util.Optional;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Component
public class InnerRoleAccessServiceServiceFacade implements RoleAccessServiceFacade {

    private final RoleAccessResponseMapper roleAccessResponseMapper;
    private final GenerateGodModAccessUseCase generateGodModAccessUseCase;
    private final FindRoleAccessesUseCase findRoleAccessesUseCase;
    private final FindRoleAccessByIdUseCase findRoleAccessByIdUseCase;
    private final CreateRoleAccessUseCase createRoleAccessUseCase;
    private final UpdateRoleAccessUseCase updateRoleAccessUseCase;
    private final DeleteRoleAccessUseCase deleteRoleAccessUseCase;

    @Override
    public RoleAccessResponse generateGodMode() {
        var result = generateGodModAccessUseCase.execute(new VoidInputValues());
        return roleAccessResponseMapper.map(result.getValue());
    }

    @Override
    public SearchResult<RoleAccessResponse> find(FindRoleAccessWithFiltersRequest command) {
        var result = findRoleAccessesUseCase.execute(
                FiltersInputValues.of(RoleAccessFilters.builder()
                        .limit(command.getLimit())
                        .offset(command.getOffset())
                        .query(command.getQuery())
                        .sortName(command.getSortName())
                        .sortDirection(command.getSortDirection())
                        .method(Optional.ofNullable(command.getMethod()).map(m -> RoleAccessEntity.Method.valueOf(m.name())).orElse(null))
                        .deleted(command.getIsDeleted())
                        .build()));
        return SearchResult.withItemsAndCount(
                result.getResult().getItems().stream().map(roleAccessResponseMapper::map).collect(Collectors.toList()),
                result.getResult().getCount()
        );
    }

    @Override
    public Optional<RoleAccessResponse> findById(String id) {
        var result = findRoleAccessByIdUseCase.execute(IdInputValues.of(id));
        return result.getEntity().map(roleAccessResponseMapper::map);
    }

    @Override
    public RoleAccessResponse create(CreateRoleAccessRequest command) {
        var result = createRoleAccessUseCase.execute(
                CreateRoleAccessUseCase.InputValues.builder()
                        .requestPathPattern(command.getRequestPathPattern())
                        .method(Optional.ofNullable(command.getMethod())
                                .map(r -> RoleAccessEntity.Method.valueOf(r.name()))
                                .orElse(null))
                        .serviceName(command.getServiceName())
                        .build());
        return roleAccessResponseMapper.map(result.getValue());
    }

    @Override
    public RoleAccessResponse update(String id, UpdateRoleAccessRequest command) {
        var result = updateRoleAccessUseCase.execute(
                UpdateRoleAccessUseCase.InputValues.builder()
                        .id(id)
                        .requestPathPattern(command.getRequestPathPattern())
                        .method(Optional.ofNullable(command.getMethod())
                                .map(r -> RoleAccessEntity.Method.valueOf(r.name()))
                                .orElse(null))
                        .serviceName(command.getServiceName())
                        .build());
        return roleAccessResponseMapper.map(result.getValue());
    }

    @Override
    public void delete(String id) {
        deleteRoleAccessUseCase.execute(IdInputValues.of(id));
    }
}
