package ru.foodtechlab.lib.auth.integration.inner.token;

import com.rcore.domain.commons.port.dto.SearchResult;
import com.rcore.domain.commons.usecase.model.FiltersInputValues;
import com.rcore.domain.commons.usecase.model.IdInputValues;
import com.rcore.domain.commons.usecase.model.SingleInput;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.integration.core.token.accessToken.AccessTokenServiceFacade;
import ru.foodtechlab.lib.auth.integration.inner.token.mapper.AccessTokenResponseMapper;
import ru.foodtechlab.lib.auth.service.domain.token.entity.RefreshTokenEntity;
import ru.foodtechlab.lib.auth.service.domain.token.port.filter.AccessTokenFilters;
import ru.foodtechlab.lib.auth.service.domain.token.usecases.*;
import ru.foodtechlab.lib.auth.service.facade.accessToken.dto.requests.AccessTokenFiltersRequest;
import ru.foodtechlab.lib.auth.service.facade.accessToken.dto.requests.DecodeTokenRequest;
import ru.foodtechlab.lib.auth.service.facade.accessToken.dto.responses.AccessTokenResponse;
import ru.foodtechlab.lib.auth.service.facade.accessToken.dto.responses.EncodedTokenResponse;

import java.util.Optional;
import java.util.stream.Collectors;

@Component
@RequiredArgsConstructor
public class InnerAccessTokenFacade implements AccessTokenServiceFacade {
    private final AccessTokenResponseMapper mapper;
    private final FindAccessTokensUseCase findAccessTokensUseCase;
    private final FindAccessTokenByIdUseCase findAccessTokenByIdUseCase;
    private final DecodeAccessTokenUseCase decodeAccessTokenUseCase;
    private final EncodeAccessTokenUseCase encodeAccessTokenUseCase;
    private final ExpireAccessTokenByStatusUseCase expireAccessTokenByStatusUseCase;
    private final DeactivateTokensByCredentialUseCase deactivateTokensByCredentialUseCase;
    private final ExpireAccessTokenByTimeUseCase expireAccessTokenByTimeUseCase;

    @Override
    public SearchResult<AccessTokenResponse> find(AccessTokenFiltersRequest request) {
        var result = findAccessTokensUseCase.execute(
                FiltersInputValues.of(AccessTokenFilters.builder()
                        .createByRefreshTokenId(request.getCreateByRefreshTokenId())
                        .credentialId(request.getCredentialId())
                        .status(Optional.ofNullable(request.getStatus())
                                .map(s -> RefreshTokenEntity.Status.valueOf(s.name()))
                                .orElse(null))
                        .limit(request.getLimit())
                        .offset(request.getOffset())
                        .query(request.getQuery())
                        .sortDirection(request.getSortDirection())
                        .sortName(request.getSortName())
                        .build()));
        return SearchResult.withItemsAndCount(
                result.getResult().getItems().stream().map(mapper::map).collect(Collectors.toList()),
                result.getResult().getCount()
        );
    }

    @Override
    public Optional<AccessTokenResponse> findById(String id) {
        var result = findAccessTokenByIdUseCase.execute(IdInputValues.of(id));
        return result.getEntity().map(mapper::map);
    }

    @Override
    public AccessTokenResponse decode(DecodeTokenRequest request) {
        var result = decodeAccessTokenUseCase.execute(SingleInput.of(request.getToken()));
        return mapper.map(result.getValue());
    }

    @Override
    public EncodedTokenResponse encode(String id) {
        var result = encodeAccessTokenUseCase.execute(SingleInput.of(id));
        return EncodedTokenResponse.of(result.getValue());
    }

    @Override
    public void expireByStatus(String id) {
        expireAccessTokenByStatusUseCase.execute(IdInputValues.of(id));
    }

    @Override
    public void deactivateByCredentialId(String credentialId) {
        deactivateTokensByCredentialUseCase.execute(DeactivateTokensByCredentialUseCase.InputValues.of(credentialId));
    }

    @Override
    public void expireByTime(String id) {
        expireAccessTokenByTimeUseCase.execute(IdInputValues.of(id));
    }
}
