package ru.foodtechlab.lib.auth.integration.inner.token;

import com.rcore.domain.commons.port.dto.SearchResult;
import com.rcore.domain.commons.usecase.model.FiltersInputValues;
import com.rcore.domain.commons.usecase.model.IdInputValues;
import com.rcore.domain.commons.usecase.model.SingleInput;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.integration.core.token.refreshToken.RefreshTokenServiceFacade;
import ru.foodtechlab.lib.auth.integration.inner.token.mapper.RefreshTokenResponseMapper;
import ru.foodtechlab.lib.auth.service.domain.token.entity.RefreshTokenEntity;
import ru.foodtechlab.lib.auth.service.domain.token.port.filter.RefreshTokenFilters;
import ru.foodtechlab.lib.auth.service.domain.token.usecases.*;
import ru.foodtechlab.lib.auth.service.facade.accessToken.dto.requests.DecodeTokenRequest;
import ru.foodtechlab.lib.auth.service.facade.accessToken.dto.responses.EncodedTokenResponse;
import ru.foodtechlab.lib.auth.service.facade.refreshToken.dto.requests.RefreshTokenFiltersRequest;
import ru.foodtechlab.lib.auth.service.facade.refreshToken.dto.responses.RefreshTokenResponse;

import java.util.Optional;
import java.util.stream.Collectors;

@Component
@RequiredArgsConstructor
public class InnerRefreshTokenFacade implements RefreshTokenServiceFacade {

    private final RefreshTokenResponseMapper mapper;
    private final FindRefreshTokensUseCase findRefreshTokensUseCase;
    private final ForceFindRefreshTokenByIdUseCase forceFindRefreshTokenByIdUseCase;
    private final DecodeRefreshTokenUseCase decodeRefreshTokenUseCase;
    private final EncodeRefreshTokenUseCase encodeRefreshTokenUseCase;
    private final ExpireRefreshTokenByStatusUseCase expireRefreshTokenByStatusUseCase;
    private final ExpireRefreshTokenByTimeUseCase expireRefreshTokenByTimeUseCase;

    @Override
    public SearchResult<RefreshTokenResponse> find(RefreshTokenFiltersRequest request) {
        var result = findRefreshTokensUseCase.execute(
                FiltersInputValues.of(RefreshTokenFilters.builder()
                        .credentialId(request.getCredentialId())
                        .status(Optional.ofNullable(request.getStatus())
                                .map(s -> RefreshTokenEntity.Status.valueOf(s.name()))
                                .orElse(null))
                        .deleted(request.getIsDeleted())
                        .limit(request.getLimit())
                        .offset(request.getOffset())
                        .query(request.getQuery())
                        .sortDirection(request.getSortDirection())
                        .sortName(request.getSortName())
                        .build()));
        return SearchResult.withItemsAndCount(
                result.getResult().getItems().stream().map(mapper::map).collect(Collectors.toList()),
                result.getResult().getCount()
        );
    }

    @Override
    public Optional<RefreshTokenResponse> findById(String id) {
        var result = forceFindRefreshTokenByIdUseCase.execute(IdInputValues.of(id));
        return result.getEntity().map(mapper::map);
    }

    @Override
    public RefreshTokenResponse decode(DecodeTokenRequest request) {
        var result = decodeRefreshTokenUseCase.execute(SingleInput.of(request.getToken()));
        return mapper.map(result.getValue());
    }

    @Override
    public EncodedTokenResponse encode(String id) {
        var result = encodeRefreshTokenUseCase.execute(SingleInput.of(id));
        return EncodedTokenResponse.of(result.getValue());
    }

    @Override
    public void expireByStatus(String id) {
        expireRefreshTokenByStatusUseCase.execute(IdInputValues.of(id));
    }

    @Override
    public void expireByTime(String id) {
        expireRefreshTokenByTimeUseCase.execute(IdInputValues.of(id));
    }
}
