package com.shsnc.shsncrocket.aspectj;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.core.util.ArrayUtil;
import com.alibaba.fastjson.JSON;
import com.shsnc.shsncrocket.core.annotation.SncLog;
import com.shsnc.shsncrocket.core.rocketmq.EnhanceRocketMQConfig;
import com.shsnc.shsncrocket.core.rocketmq.RocketMQEnhanceTemplate;
import com.shsnc.shsncrocket.core.rocketmq.message.SncOperLogMessage;
import com.shsnc.shsncrocket.entiy.SncOperLog;
import com.shsnc.shsncrocket.util.ServletUtils;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpMethod;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindingResult;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Map;

/**
 * 操作日志记录处理
 */
@Component
@Slf4j
public class SncOperLogAspect implements MethodInterceptor {

    private RocketMQEnhanceTemplate rocketMQEnhanceTemplate;

    public SncOperLogAspect(RocketMQEnhanceTemplate rocketMQEnhanceTemplate) {
        this.rocketMQEnhanceTemplate = rocketMQEnhanceTemplate;
    }

    @Nullable
    @Override
    public Object invoke(@Nonnull MethodInvocation invocation) throws Throwable {
        long startTime = System.currentTimeMillis();
        try {
            Object result = invocation.proceed();
            long endTime = System.currentTimeMillis();
            long costTime = endTime - startTime;
            HttpServletRequest request = ServletUtils.getRequest();
            ThreadUtil.execute(() -> {
                handleLog(invocation, result, costTime, null, request);
            });
            return result;
        } catch (Exception e) {
            long endTime = System.currentTimeMillis();
            long costTime = endTime - startTime;
            Method method = invocation.getMethod();
            if (method.isAnnotationPresent(Scheduled.class)) {
                return null;
            }
            HttpServletRequest request = ServletUtils.getRequest();
            ThreadUtil.execute(() -> {
                handleLog(invocation, null, costTime, e, request);
            });
            throw e;
        }
    }

    /**
     * 处理完请求后执行
     *
     * @param invocation 切点
     * @param result     返回结果
     * @param cost       耗时
     * @param e          异常
     * @param request
     */
    private void handleLog(MethodInvocation invocation, Object result, long cost, Exception e, HttpServletRequest request) {
        try {
            SncOperLog sncOperLog = new SncOperLog();
            if (e != null) {
                sncOperLog.setStatus(0);//失败
                sncOperLog.setErrorMsg(StringUtils.substring(e.getMessage(), 0, 2000));
            } else {
                sncOperLog.setStatus(1);//成功
            }
            String clientIP = ServletUtils.getClientIP(request);
            sncOperLog.setOperIp(clientIP);
            sncOperLog.setOperUrl(request.getRequestURI());
            sncOperLog.setOperTime(LocalDateTime.now());
            //reflectiveMethodInvocation.getMethod().getName():方法名
            Method method = invocation.getMethod();
            sncOperLog.setMethod(method.getDeclaringClass().getName() + "." + method.getName() + "()");
            sncOperLog.setRequestMethod(request.getMethod());
            //标题
            SncLog sncLog = method.getAnnotation(SncLog.class);
            if (sncLog != null) {
                sncOperLog.setTitle(sncLog.title());
            }
            //入参
            setRequestValue(invocation, sncOperLog, request);
            //返回值
            if (result != null) {
                sncOperLog.setJsonResult(StringUtils.substring(JSON.toJSONString(result), 0, 2000));
            }
            sncOperLog.setCost(cost);
            //发送日志到mq
            SncOperLogMessage<SncOperLog> sncOperLogMessage = new SncOperLogMessage<>();
            sncOperLogMessage.setSncOperLog(sncOperLog);
            sncOperLogMessage.setSystemName(EnhanceRocketMQConfig.getSystemName());
            rocketMQEnhanceTemplate.send(EnhanceRocketMQConfig.getOperLogTopicName(), sncOperLogMessage);
        } catch (Exception exp) {
            // 记录本地异常日志
            log.error("==环绕通知异常==");
            log.error("异常信息:{}", exp.getMessage());
            exp.printStackTrace();
        }
    }

    /**
     * 获取请求的参数，放到log中
     *
     * @param operLog 操作日志
     * @param request
     */
    private void setRequestValue(MethodInvocation invocation, SncOperLog operLog, HttpServletRequest request) {
        Map<?, ?> paramsMap = ServletUtils.getParamMap(request);
        String requestMethod = operLog.getRequestMethod();
        if (MapUtil.isEmpty(paramsMap) && (HttpMethod.PUT.name().equals(requestMethod) || HttpMethod.POST.name().equals(requestMethod))) {
            String params = argsArrayToString(invocation.getArguments());
            operLog.setOperParam(StringUtils.substring(params, 0, 2000));
        } else {
            operLog.setOperParam(StringUtils.substring(paramsMap.toString(), 0, 2000));
        }
    }

    /**
     * 参数拼装
     */
    private String argsArrayToString(Object[] paramsArray) {
        StringBuilder params = new StringBuilder();
        if (ArrayUtil.isNotEmpty(paramsArray)) {
            for (Object o : paramsArray) {
                if (o != null && !isFilterObject(o)) {
                    try {
                        Object jsonObj = JSON.toJSON(o);
                        params.append(jsonObj.toString()).append(" ");
                    } catch (Exception ignored) {
                    }
                }
            }
        }
        return params.toString().trim();
    }


    public boolean isFilterObject(final Object o) {
        Class<?> clazz = o.getClass();
        if (clazz.isArray()) {
            return clazz.getComponentType().isAssignableFrom(MultipartFile.class);
        } else if (Collection.class.isAssignableFrom(clazz)) {
            Collection collection = (Collection) o;
            for (Object value : collection) {
                return value instanceof MultipartFile;
            }
        } else if (Map.class.isAssignableFrom(clazz)) {
            Map map = (Map) o;
            for (Object value : map.entrySet()) {
                Map.Entry entry = (Map.Entry) value;
                return entry.getValue() instanceof MultipartFile;
            }
        }
        return o instanceof MultipartFile || o instanceof HttpServletRequest || o instanceof HttpServletResponse || o instanceof BindingResult;
    }
}
