package io.gitee.declear.dec.cloud.common.config;

import io.gitee.declear.common.utils.CommonUtils;
import io.gitee.declear.dec.cloud.common.annotation.DecCloudInBound;
import io.gitee.declear.dec.cloud.common.annotation.DecCloudOutBound;
import io.gitee.declear.dec.cloud.common.annotation.DecCloudService;
import io.gitee.declear.dec.cloud.common.config.spring.DecCloudServiceBeanFactory;
import io.gitee.declear.dec.cloud.common.config.spring.web.DecHttpRequestObjectFactory;
import io.gitee.declear.dec.cloud.common.config.spring.web.DecHttpResponseObjectFactory;
import io.gitee.declear.dec.cloud.common.config.spring.web.DecHttpSessionObjectFactory;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.exception.DecCloudServiceException;
import io.gitee.declear.dec.cloud.common.property.PropertiesManager;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContext;
import io.gitee.declear.dec.cloud.common.remoting.invoke.DecCloudInvoker;
import io.gitee.declear.dec.cloud.common.remoting.resource.DecCloudApi;
import io.gitee.declear.dec.cloud.common.remoting.resource.DecCloudResource;
import io.gitee.declear.dec.cloud.common.remoting.resource.DecCloudResourceManager;
import io.gitee.declear.dec.cloud.common.rpc.netty.NettyServer;
import io.gitee.declear.dec.cloud.common.web.DecHttpRequest;
import io.gitee.declear.dec.cloud.common.web.DecHttpResponse;
import io.gitee.declear.dec.cloud.common.web.DecHttpSession;
import io.gitee.declear.dec.cloud.common.web.bind.annotation.RequestMapping;
import io.gitee.declear.dec.cloud.common.web.bind.annotation.RestController;
import io.gitee.declear.dec.cloud.common.web.context.DecWebContextManager;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ResourceLoaderAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternUtils;
import org.springframework.core.type.classreading.CachingMetadataReaderFactory;
import org.springframework.core.type.classreading.MetadataReader;
import org.springframework.core.type.classreading.MetadataReaderFactory;
import org.springframework.util.ClassUtils;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * DecCloud框架相关的注解处理，解析接口，启动netty，服务注册，接口发布
 * @author DEC
 */
@Slf4j
@Configuration
public class DecCloudDefinitionRegistry implements BeanDefinitionRegistryPostProcessor, ResourceLoaderAware, ApplicationContextAware, BeanPostProcessor {

    private static final String PATTERN_ALL_CLASS = "/**/*.class";

    private ApplicationContext applicationContext;

    private ResourcePatternResolver resourcePatternResolver;

    private MetadataReaderFactory metadataReaderFactory;

    private final Map<String, DecCloudApi> decOutBoundInterfaceMap = new ConcurrentHashMap<>();
    private final Map<String, DecCloudApi> decInBoundInterfaceMap = new ConcurrentHashMap<>();
    private final Map<String, Method> restUriMap = new ConcurrentHashMap<>();

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        List<String> basePackages = AutoConfigurationPackages.get(applicationContext);
        basePackages.forEach(basePackage -> {

            Set<Class<?>> classSet = scanPackages(basePackage);
            for (Class<?> clazz : classSet) {

                // 全局扫描@DecCloudService
                if(isContainAnnotationsOfDecCloudService(clazz)) {
                    BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(clazz);
                    GenericBeanDefinition definition = (GenericBeanDefinition) builder.getRawBeanDefinition();

                    definition.setBeanClass(DecCloudServiceBeanFactory.class);
                    definition.getConstructorArgumentValues().addGenericArgumentValue(clazz);
                    definition.getConstructorArgumentValues().addGenericArgumentValue(applicationContext);
                    definition.setAutowireMode(GenericBeanDefinition.AUTOWIRE_BY_TYPE);

                    registry.registerBeanDefinition(generateBeanName(clazz), definition);

                    // 处理 @DecCloudInBound 注解
                    analyzeDecCloudInBoundAnnotation(clazz);
                }

                // 处理 @DecCloudOutBound 注解
                analyzeDecCloudOutBoundAnnotation(clazz);

                // 处理 @RestController web请求相关 注解
                if(isContainAnnotationsOfRestController(clazz)) {
                    BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(clazz);
                    GenericBeanDefinition definition = (GenericBeanDefinition) builder.getRawBeanDefinition();

                    definition.setBeanClass(clazz);
                    definition.setAutowireMode(GenericBeanDefinition.AUTOWIRE_BY_TYPE);
                    definition.setScope(BeanDefinition.SCOPE_PROTOTYPE);

                    registry.registerBeanDefinition(generateBeanName(clazz), definition);

                    analyzeDecCloudRestControllerAnnotation(clazz);
                }
            }
        });
    }

    private void analyzeDecCloudOutBoundAnnotation(Class<?> clazz) {
        Method[] methods = clazz.getMethods();
        if(methods.length > 0) {
            for (Method method : methods) {
                DecCloudOutBound decCloudOutBound = method.getDeclaredAnnotation(DecCloudOutBound.class);
                if(decCloudOutBound != null) {
                    DecCloudApi decCloudApi = new DecCloudApi();
                    decCloudApi.setBeanClass(clazz);
                    decCloudApi.setInterfaceMethod(method.getName());
                    decCloudApi.setParameterTypes(method.getParameterTypes());
                    decOutBoundInterfaceMap.put(decCloudApi.generateIdWithClass(), decCloudApi);
                }
            }
        }
    }

    private void analyzeDecCloudInBoundAnnotation(Class<?> clazz) {
        Method[] methods = clazz.getMethods();
        if(methods.length > 0) {
            DecCloudService decCloudService = clazz.getAnnotation(DecCloudService.class);
            for (Method method : methods) {
                DecCloudInBound decCloudInBound = method.getDeclaredAnnotation(DecCloudInBound.class);
                if(decCloudInBound != null) {
                    DecCloudApi decCloudApi = new DecCloudApi();
                    decCloudApi.setInstance(decCloudService.instance());
                    decCloudApi.setBeanClass(clazz);
                    decCloudApi.setInterfaceMethod(method.getName());
                    decCloudApi.setParameterTypes(method.getParameterTypes());
                    decInBoundInterfaceMap.put(decCloudApi.generateIdWithInstanceAndMethod(), decCloudApi);
                }
            }
        }
    }

    private void analyzeDecCloudRestControllerAnnotation(Class<?> clazz) {
        Method[] methods = clazz.getMethods();
        if(methods.length > 0) {
            PropertiesManager propertiesManager = applicationContext.getBean(PropertiesManager.class);
            RestController restController = clazz.getAnnotation(RestController.class);
            for (Method method : methods) {
                RequestMapping requestMapping = method.getDeclaredAnnotation(RequestMapping.class);
                if(requestMapping != null) {
                    StringBuilder uriBuilder = new StringBuilder(requestMapping.method().name()).append("#");
                    String contextPath = propertiesManager.getProperty(Constants.DEC_CLOUD_CONTEXT_PATH);
                    if(CommonUtils.isNotEmpty(contextPath)) {
                        if (contextPath.charAt(0) != Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER) {
                            uriBuilder.append(Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER);
                        }
                        if (contextPath.charAt(contextPath.length() - 1) == Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER) {
                            uriBuilder.append(contextPath, 0, contextPath.length() - 1);
                        } else {
                            uriBuilder.append(contextPath);
                        }
                    }
                    if(restController.path().charAt(0) != Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER) {
                        uriBuilder.append(Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER);
                    }
                    if(restController.path().charAt(restController.path().length() - 1) == Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER) {
                        uriBuilder.append(restController.path(), 0, restController.path().length() - 1);
                    } else {
                        uriBuilder.append(restController.path());
                    }

                    if(requestMapping.path().charAt(0) != Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER) {
                        uriBuilder.append(Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER);
                    }
                    if(requestMapping.path().charAt(requestMapping.path().length() - 1) == Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER) {
                        uriBuilder.append(requestMapping.path(), 0, requestMapping.path().length() - 1);
                    } else {
                        uriBuilder.append(requestMapping.path());
                    }

                    restUriMap.put(uriBuilder.toString(), method);
                }
            }
        }
    }

    private String generateBeanName(Class<?> clazz) {
        String beanName = clazz.getSimpleName();
        return beanName.substring(0, 1).toLowerCase() + beanName.substring(1);
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        switch (beanName) {
            case "decCloudResourceManager":
                DecCloudResourceManager decCloudResourceManager = (DecCloudResourceManager) bean;
                decCloudResourceManager.putDecOutBoundInterfaces(decOutBoundInterfaceMap);
                decCloudResourceManager.putDecInBoundInterfaces(decInBoundInterfaceMap);
                break;
            case "decWebContextManager":
                DecWebContextManager decWebContextManager = (DecWebContextManager) bean;
                decWebContextManager.putRestControllerMap(restUriMap);
                break;
            case "nettyServer":
                NettyServer nettyServer = (NettyServer) bean;
                initNettyServer(nettyServer);
                break;
            case "nettyClient":
                initCloudService();
                break;
            default:
                break;
        }

        return bean;
    }

    /**
     * 启动nettyServer, 如果本实例的类型为 index 则将本服务加入DecCloudResourceManager#indexList
     * @param nettyServer
     */
    private void initNettyServer(NettyServer nettyServer) {
        PropertiesManager propertiesManager = applicationContext.getBean(PropertiesManager.class);
        DecCloudResourceManager cloudResourceManager = applicationContext.getBean(DecCloudResourceManager.class);

        String cloudType = cloudResourceManager.getCloudType();

        try {
            String port = propertiesManager.getProperty(Constants.DEC_CLOUD_PORT);
            nettyServer.start(Integer.parseInt(port));

            if(Objects.equals(Constants.DEC_CLOUD_TYPE_INDEX, cloudType)) {
                // 把其他index节点添加到 DecCloudResourceManager.indexList
                String indexUrlStr = propertiesManager.getProperty(Constants.DEC_CLOUD_INDEX_URL);
                if(CommonUtils.isNotEmpty(indexUrlStr)) {
                    String[] indexUrls = indexUrlStr.split(",");
                    for (String indexUrl : indexUrls) {
                        String[] index = indexUrl.trim().split(":");
                        InetSocketAddress address = new InetSocketAddress(index[0], Integer.parseInt(index[1]));

                        DecCloudResource indexResource = new DecCloudResource();
                        indexResource.setAddress(address);
                        indexResource.setType(Constants.DEC_CLOUD_TYPE_INDEX);
                        cloudResourceManager.putIndexCloudResource(indexResource);
                    }
                }
            }
        } catch (InterruptedException | IOException e) {
            throw new DecCloudServiceException(e);
        }
    }

    /**
     * 当本实例类型为service/gateway时，注册到index
     * 实例类型为service时将本实例的对外接口发布到index
     */
    private void initCloudService() {
        DecCloudResourceManager cloudResourceManager = applicationContext.getBean(DecCloudResourceManager.class);
        PropertiesManager propertiesManager = applicationContext.getBean(PropertiesManager.class);
        DecCloudInvoker invoker = applicationContext.getBean(DecCloudInvoker.class);

        String cloudType = cloudResourceManager.getCloudType();

        if(Objects.equals(Constants.DEC_CLOUD_TYPE_SERVICE, cloudType)
                || Objects.equals(Constants.DEC_CLOUD_TYPE_GATEWAY, cloudType)) {
            // 多个indexUrl以逗号分隔
            String indexUrlStr = propertiesManager.getProperty(Constants.DEC_CLOUD_INDEX_URL);
            if(CommonUtils.isNotEmpty(indexUrlStr)) {
                String[] indexUrls = indexUrlStr.split(",");
                for (String indexUrl : indexUrls) {
                    String[] index = indexUrl.trim().split(":");
                    InetSocketAddress address = new InetSocketAddress(index[0], Integer.parseInt(index[1]));
                    DecCloudApi cloudDestination = new DecCloudApi();
                    cloudDestination.setAddress(address);

                    // 注册服务
                    DecRemoteContext<Serializable> registerContext = new DecRemoteContext<>();
                    registerContext.setId(CommonUtils.UUID());
                    registerContext.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_4);
                    registerContext.setCloudDestination(cloudDestination);
                    registerContext.setCloudOrigin(cloudResourceManager.getCloudOrigin());
                    List<Serializable> paramList = new ArrayList<>();
                    paramList.add(cloudType);
                    registerContext.setParamList(paramList);

                    invoker.invoke(registerContext, (ctx) -> {
                        // 保存所有index cloud resource
                        DecCloudResource indexResource = new DecCloudResource();
                        indexResource.setAddress(address);
                        indexResource.setType(Constants.DEC_CLOUD_TYPE_INDEX);
                        cloudResourceManager.putIndexCloudResource(indexResource);

                        if(Objects.equals(Constants.DEC_CLOUD_TYPE_SERVICE, cloudType)) {
                            // 发布本服务接口
                            if (CommonUtils.isNotEmpty(decOutBoundInterfaceMap)) {
                                DecCloudApi cloudOrigin = cloudResourceManager.getCloudOrigin();
                                decOutBoundInterfaceMap.forEach((key, value) -> {
                                    DecRemoteContext<Serializable> announceContext = new DecRemoteContext<>();
                                    announceContext.setId(CommonUtils.UUID());
                                    announceContext.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_6);
                                    value.setAddress(cloudOrigin.getAddress());
                                    value.setInstance(cloudOrigin.getInstance());
                                    announceContext.setCloudDestination(cloudDestination);
                                    announceContext.setCloudOrigin(value);

                                    invoker.invoke(announceContext);
                                });

                                // 发布接口完毕
                                DecRemoteContext<Serializable> announceCompleteContext = new DecRemoteContext<>();
                                announceCompleteContext.setId(CommonUtils.UUID());
                                announceCompleteContext.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_8);
                                announceCompleteContext.setCloudDestination(cloudDestination);
                                announceCompleteContext.setCloudOrigin(cloudResourceManager.getCloudOrigin());

                                invoker.invoke(announceCompleteContext);
                            }
                        }
                    }, null);
                }
            }
        }
    }

    private Set<Class<?>> scanPackages(String basePackage) {
        Set<Class<?>> classSet = new LinkedHashSet<>();
        String path = ResourcePatternResolver.CLASSPATH_ALL_URL_PREFIX
                + ClassUtils.convertClassNameToResourcePath(applicationContext.getEnvironment().resolveRequiredPlaceholders(basePackage))
                + PATTERN_ALL_CLASS;

        try {
            Resource[] resources = resourcePatternResolver.getResources(path);
            for (Resource resource: resources) {
                if(resource.isReadable()) {
                    MetadataReader metadataReader = metadataReaderFactory.getMetadataReader(resource);
                    String className = metadataReader.getClassMetadata().getClassName();
                    Class<?> clazz = ClassUtils.forName(className, applicationContext.getClassLoader());
                    classSet.add(clazz);
                }
            }
        } catch (IOException e) {
            log.error("process dec-cloud bean definition error.", e);
        } catch (ClassNotFoundException e) {
            log.error("process dec-cloud bean definition error: class not found.", e);
        }

        return classSet;
    }

    private Boolean isContainAnnotationsOfDecCloudService(Class<?> clazz) {
        return clazz.getAnnotation(DecCloudService.class) != null;
    }

    private Boolean isContainAnnotationsOfRestController(Class<?> clazz) {
        return clazz.getAnnotation(RestController.class) != null;
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        beanFactory.registerResolvableDependency(DecHttpRequest.class, new DecHttpRequestObjectFactory());
        beanFactory.registerResolvableDependency(DecHttpResponse.class, new DecHttpResponseObjectFactory());
        beanFactory.registerResolvableDependency(DecHttpSession.class, new DecHttpSessionObjectFactory());
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void setResourceLoader(ResourceLoader resourceLoader) {
        resourcePatternResolver = ResourcePatternUtils.getResourcePatternResolver(resourceLoader);
        metadataReaderFactory = new CachingMetadataReaderFactory(resourceLoader);
    }

}
