package io.gitee.declear.dec.cloud.common.property;

import io.gitee.declear.common.utils.NamedThreadFactory;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import oshi.SystemInfo;
import oshi.hardware.CentralProcessor;
import oshi.hardware.GlobalMemory;

import java.util.concurrent.*;

/**
 * 服务所在的实例的硬件信息，包括CPU 内存 存储容量等信息
 * @author DEC
 */
@Slf4j
@Data
public class HardWareInfoManager {

    /**
     * cpu核数
     */
    private Integer cpuCoreCount;

    /**
     * cpu占用率
     */
    private Float cpuOccupyRate;

    /**
     * cpu占用率
     */
    private Float cpuIoWaitRate;

    /**
     * 内存占用率
     */
    private Float memoryOccupyRate;

    /**
     * 内存余量
     */
    private Long memoryRemain;

    private ScheduledExecutorService schedule;

    public void init() {
        cpuCoreCount = 0;
        cpuOccupyRate = 0f;
        cpuIoWaitRate = 0f;
        memoryOccupyRate = 0f;
        memoryRemain = 0L;

        // schedule-pool-h
        schedule = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("schedule-pool-h"));
        schedule.scheduleWithFixedDelay(() -> {
            try {
                processData();
            } catch (InterruptedException e) {
                log.error("get hardware information error.", e);
            }
        }, 1,2, TimeUnit.SECONDS);
    }

    public void shutdown() {
        if(schedule != null) {
            schedule.shutdown();
        }
    }

    private void processData() throws InterruptedException {
        SystemInfo systemInfo = new SystemInfo();
        getCpuInfo(systemInfo);
        getMemoryInfo(systemInfo);
    }

    private void getCpuInfo(SystemInfo systemInfo) throws InterruptedException {
        CentralProcessor processor = systemInfo.getHardware().getProcessor();
        long[]prevTicks = processor.getSystemCpuLoadTicks();
        TimeUnit.SECONDS.sleep(1);
        long[]ticks = processor.getSystemCpuLoadTicks();
        long nice = ticks[CentralProcessor.TickType.NICE.getIndex()] - prevTicks[CentralProcessor.TickType.NICE.getIndex()];
        long irq = ticks[CentralProcessor.TickType.IRQ.getIndex()] - prevTicks[CentralProcessor.TickType.IRQ.getIndex()];
        long softIrq = ticks[CentralProcessor.TickType.SOFTIRQ.getIndex()] - prevTicks[CentralProcessor.TickType.SOFTIRQ.getIndex()];
        long steal = ticks[CentralProcessor.TickType.STEAL.getIndex()] - prevTicks[CentralProcessor.TickType.STEAL.getIndex()];
        long cSys = ticks[CentralProcessor.TickType.SYSTEM.getIndex()] - prevTicks[CentralProcessor.TickType.SYSTEM.getIndex()];
        long user = ticks[CentralProcessor.TickType.USER.getIndex()] - prevTicks[CentralProcessor.TickType.USER.getIndex()];
        long ioWait = ticks[CentralProcessor.TickType.IOWAIT.getIndex()] - prevTicks[CentralProcessor.TickType.IOWAIT.getIndex()];
        long idle = ticks[CentralProcessor.TickType.IDLE.getIndex()] - prevTicks[CentralProcessor.TickType.IDLE.getIndex()];
        long totalCpu = user + nice + cSys + idle + ioWait + irq + softIrq + steal;

        cpuCoreCount = processor.getLogicalProcessorCount();
        cpuOccupyRate = (cpuOccupyRate * 6 + (100.0f - (idle * 100.0f / totalCpu))) / 7;
        cpuIoWaitRate = ioWait * 100.0f / totalCpu;
    }

    private void getMemoryInfo(SystemInfo systemInfo) {
        GlobalMemory memory = systemInfo.getHardware().getMemory();

        memoryRemain = memory.getAvailable();
        memoryOccupyRate = 100.0f - memoryRemain * 100.0f / memory.getTotal();
    }

}
