package io.gitee.declear.dec.cloud.common.remoting;

import io.gitee.declear.common.utils.CommonUtils;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.property.HardWareInfoManager;
import io.gitee.declear.dec.cloud.common.remoting.invoke.DecCloudInvoker;
import io.gitee.declear.dec.cloud.common.remoting.resource.DecCloudApi;
import io.gitee.declear.dec.cloud.common.remoting.resource.DecCloudResource;
import io.gitee.declear.dec.cloud.common.remoting.resource.DecCloudResourceManager;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.io.Serializable;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.*;

/**
 * Dec cloud框架相关的所有资源的管理类
 * @author DEC
 */
@Slf4j
@Data
public class DecRemoteContextManager implements ApplicationContextAware {

    @Autowired
    private HardWareInfoManager hardWareInfoManager;

    @Autowired
    private DecCloudResourceManager decCloudResourceManager;

    /**
     * 发送端发送到远端，等待回应的DecRemoteContext的map
     */
    private Map<String, DecRemoteContext> holdContextMap;

    private DecRemoteContextProcessor contextProcessor;

    private ApplicationContext applicationContext;

    /**
     * service/gateway 需要报告给 index 的报错信息， 缓存
     */
    private List<RuntimeException> runtimeExceptionList;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    public void init() {
        holdContextMap = new ConcurrentHashMap<>(10000);

        contextProcessor = new DecRemoteContextProcessor(applicationContext, hardWareInfoManager);
    }

    public void shutdown() {
        if(contextProcessor != null) {
            contextProcessor.shutdown();
        }
    }

    public void putHoldRemoteContext(DecRemoteContext<Serializable> context) {
        holdContextMap.put(context.getId(), context);
        DecRemoteContextHolder.setRemoteContext(context);
    }

    public DecRemoteContext<Serializable> getHoldRemoteContext(String id) {
        return holdContextMap.get(id);
    }

    public void removeHoldRemoteContext(String id) {
        holdContextMap.remove(id);
    }

    public void pushException2Index(RuntimeException exception) {
        if(Objects.equals(Constants.DEC_CLOUD_TYPE_SERVICE, decCloudResourceManager.getCloudType())
                || Objects.equals(Constants.DEC_CLOUD_TYPE_GATEWAY, decCloudResourceManager.getCloudType())) {
            if (null != decCloudResourceManager.getMainIndex()) {
                doPushException2Index(exception);
            } else {
                runtimeExceptionList.add(exception);
            }
        }
    }

    public void pushAllException2Index() {
        if(Objects.equals(Constants.DEC_CLOUD_TYPE_SERVICE, decCloudResourceManager.getCloudType())
                || Objects.equals(Constants.DEC_CLOUD_TYPE_GATEWAY, decCloudResourceManager.getCloudType())) {
            if(CommonUtils.isNotEmpty(runtimeExceptionList)) {
                runtimeExceptionList.forEach(exception -> {
                    doPushException2Index(exception);
                });
            }
        }
    }

    private void doPushException2Index(Exception exception) {
        DecCloudInvoker invoker = applicationContext.getBean(DecCloudInvoker.class);
        DecRemoteContext<Serializable> context = new DecRemoteContext<>();
        context.setId(CommonUtils.UUID());
        context.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_12);
        DecCloudApi cloudDestination = new DecCloudApi();
        cloudDestination.setAddress(decCloudResourceManager.getMainIndex().getAddress());
        cloudDestination.setInstance(decCloudResourceManager.getMainIndex().getInstance());
        context.setCloudDestination(cloudDestination);
        context.setCloudOrigin(decCloudResourceManager.getCloudOrigin());
        List<Serializable> paramList = new ArrayList<>();
        paramList.add(exception.toString());
        context.setParamList(paramList);

        invoker.invoke(context);
    }

    public void putProcessRemoteContext(DecRemoteContext<Serializable> context) {
        switch (context.getType()) {
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_1 -> contextProcessor.processDecRemoteContext(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_2 -> processRemoteContextBack(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_4 -> processServiceRegister(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_5 -> processServiceRegisterBack(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_6 -> processServiceAnnounceRemoteApi(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_7 -> processServiceAnnounceRemoteApiBack(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_8 -> processServiceAnnounceRemoteApiComplete(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_9 -> processServiceAnnounceRemoteApiCompleteBack(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_10 -> processRecruit(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_11 -> processRecruitBack(context);
            case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_12 -> processException(context);
            default -> throw new IllegalStateException("unsupport context type: " + context.getType());
        }
    }

    /**
     * service发布的远程调用接口完毕信号返回结果
     * @param context
     */
    private void processServiceAnnounceRemoteApiCompleteBack(DecRemoteContext<Serializable> context) {
        String instance = decCloudResourceManager.getCloudOrigin().getInstance();
        if(Objects.equals(context.getBackStatus(), DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS)) {
            log.info("service '{}' announce api complete.", instance);
        } else {
            log.error("service '{}' announce api complete error.", instance, context.getFailure());
        }

        holdContextMap.remove(context.getId());
    }

    /**
     * index服务接收service发布的远程调用接口完毕信号
     * @param context
     */
    private void processServiceAnnounceRemoteApiComplete(DecRemoteContext<Serializable> context) {
        try {
            Map<InetSocketAddress, DecCloudResource> cloudResourceMap = decCloudResourceManager.getServiceInstanceMap().get(context.getCloudOrigin().getInstance());
            cloudResourceMap.get(context.getCloudOrigin().getAddress()).setIsApiComplete(true);
            log.info("service '{}#{}' announce api complete.", context.getCloudOrigin().getAddress(), context.getCloudOrigin().getInstance());
            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS);
        } catch (Exception e) {
            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_FAILURE);
            context.setFailure(e);
        }

        // 服务注册成功, 返回信息
        DecCloudInvoker invoker = applicationContext.getBean(DecCloudInvoker.class);
        context.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_9);
        invoker.back(context);
    }

    /**
     * service发布远程调用接口的返回结果
     * @param context
     */
    private void processServiceAnnounceRemoteApiBack(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> holdContext = holdContextMap.get(context.getId());
        if(Objects.equals(context.getBackStatus(), DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS)) {
            log.info("out bound service interface '{}' announce success.", holdContext.getCloudOrigin());
        } else {
            log.error("out bound service interface '{}' announce error.", holdContext.getCloudOrigin(), context.getFailure());
        }

        holdContextMap.remove(context.getId());
    }

    /**
     * index服务接收service发布的远程调用接口
     * @param context
     */
    private void processServiceAnnounceRemoteApi(DecRemoteContext<Serializable> context) {
        try {
            decCloudResourceManager.putServiceCloudApi(context.getCloudOrigin());

            log.info("out bound service interface '{}' announce success.", context.getCloudOrigin());
            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS);
        } catch (Exception e) {
            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_FAILURE);
            context.setFailure(e);
        }

        // 发布本服务接口成功, 返回信息
        DecCloudInvoker invoker1 = applicationContext.getBean(DecCloudInvoker.class);
        context.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_7);
        invoker1.back(context);
    }

    /**
     * 服务处理服务注册返回结果
     * @param context
     */
    private void processServiceRegisterBack(DecRemoteContext<Serializable> context) {
        String instance = decCloudResourceManager.getCloudOrigin().getInstance();
        if(Objects.equals(context.getBackStatus(), DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS)) {
            log.info("service '{}' register success.", instance);
        } else {
            log.error("service '{}' register error.", instance, context.getFailure());
        }

        holdContextMap.remove(context.getId());
    }

    /**
     * index服务处理服务注册
     * @param context
     */
    private void processServiceRegister(DecRemoteContext<Serializable> context) {
        try {
            DecCloudResource serviceResource = new DecCloudResource();
            serviceResource.setAddress(context.getCloudOrigin().getAddress());
            serviceResource.setInstance(context.getCloudOrigin().getInstance());
            serviceResource.setType(context.getParamList().get(0).toString());
            switch (serviceResource.getType()) {
                case Constants.DEC_CLOUD_TYPE_SERVICE -> decCloudResourceManager.putServiceCloudResource(serviceResource);
                case Constants.DEC_CLOUD_TYPE_GATEWAY -> decCloudResourceManager.putGatewayCloudResource(serviceResource);
                default -> throw new IllegalStateException("unsupport dec cloud type: " + serviceResource.getType());
            }

            log.info("service '{}#{}' register success.", context.getCloudOrigin().getAddress(), context.getCloudOrigin().getInstance());
            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS);
        } catch (Exception e) {
            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_FAILURE);
            context.setFailure(e);
        }

        // 服务注册成功, 返回信息
        DecCloudInvoker invoker = applicationContext.getBean(DecCloudInvoker.class);
        context.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_5);
        invoker.back(context);
    }

    /**
     * 处理远程调用返回的结果
     * @param context
     */
    private void processRemoteContextBack(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> holdContext = holdContextMap.get(context.getId());
        holdContext.setBackStatus(context.getBackStatus());
        holdContext.setResult(context.getResult());
        holdContext.setFailure(context.getFailure());
        holdContext.setCompleteStatus(DecRemoteContext.REMOTE_CONTEXT_COMPLETE_STATUS_COMPLETE);

        if(Objects.equals(context.getBackStatus(), DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS)) {
            holdContext.getOnSuccess().exec(holdContext.getResult());
            holdContext.getOnComplete().forEach(onComplete -> {
                ((DecPromise<Serializable>) onComplete).exec(holdContext.getResult());
            });
        } else {
            holdContext.getOnFailure().exec(holdContext.getFailure());
            holdContext.getOnComplete().forEach(onComplete -> {
                ((DecPromise<Serializable>) onComplete).exec(Constants.DEC_CLOUD_WEB_SERVER_INTERNAL_ERROR);
            });
        }

        holdContextMap.remove(context.getId());
    }

    /**
     * 处理服务间建立联系
     * @param context
     */
    private void processRecruit(DecRemoteContext<Serializable> context) {
        decCloudResourceManager.getRecruitProcessor().process(context);
    }

    /**
     * 处理服务间建立联系返回的结果
     * @param context
     */
    private void processRecruitBack(DecRemoteContext<Serializable> context) {
        decCloudResourceManager.getRecruitProcessor().back(context);
    }

    /**
     * 处理 service/gateway 上报的 exception
     * @param context
     */
    private void processException(DecRemoteContext<Serializable> context) {
        if(Objects.equals(Constants.DEC_CLOUD_TYPE_INDEX, decCloudResourceManager.getCloudType())) {

        }
    }

}
