package io.gitee.declear.dec.cloud.common.remoting;

import io.gitee.declear.common.utils.NamedThreadFactory;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.property.HardWareInfoManager;
import io.gitee.declear.dec.cloud.common.remoting.resource.DecCloudApi;
import io.gitee.declear.dec.cloud.common.remoting.invoke.DecCloudInvoker;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.concurrent.*;

/**
 * DecRemoteContext Processor
 * @author DEC
 */
@Slf4j
public class DecRemoteContextProcessor {

    private ApplicationContext applicationContext;

    private HardWareInfoManager hardWareInfoManager;

    /**
     * 处理DecRemoteContext的线程池
     */
    private ThreadPoolExecutor threadPool;

    /**
     * 管理线程池的核心线程数的定时任务
     */
    private ScheduledExecutorService coreSizeSchedule;

    private Integer initThreadPoolCoreSize;

    private static final Integer THREAD_POOL_CORE_SIZE_INCREMENT = 3;
    private static final Integer THREAD_POOL_CORE_SIZE_INCREMENT_TWOFOLD = 6;
    private static final Float CPU_AND_MEMORY_BUSY_THRESHOLD = 90.0f;

    public DecRemoteContextProcessor(ApplicationContext applicationContext, HardWareInfoManager hardWareInfoManager) {
        this.applicationContext = applicationContext;
        this.hardWareInfoManager = hardWareInfoManager;

        initThreadPoolCoreSize = 4 * hardWareInfoManager.getCpuCoreCount() + 1;
        ThreadFactory namedThreadFactory = new NamedThreadFactory("r-context-pool-");
        threadPool = new ThreadPoolExecutor(initThreadPoolCoreSize, Integer.MAX_VALUE, 60L, TimeUnit.SECONDS,
                new SynchronousQueue<>(), namedThreadFactory);

        // schedule-pool-cp
        coreSizeSchedule = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("schedule-pool-cp-r"));
        coreSizeSchedule.scheduleWithFixedDelay(() -> {
            manageThreadPoolSize();
        }, 10,5, TimeUnit.SECONDS);
    }

    private void manageThreadPoolSize() {
        if(threadPool.getQueue().size() > 0 && hardWareInfoManager.getCpuOccupyRate() < CPU_AND_MEMORY_BUSY_THRESHOLD && hardWareInfoManager.getMemoryOccupyRate() < CPU_AND_MEMORY_BUSY_THRESHOLD) {
            // 线程池排队多时，增加核心线程
            threadPool.setCorePoolSize(threadPool.getCorePoolSize() + THREAD_POOL_CORE_SIZE_INCREMENT);
        } else if(threadPool.getCorePoolSize() > initThreadPoolCoreSize && threadPool.getQueue().size() == 0 && threadPool.getActiveCount() < threadPool.getCorePoolSize() - THREAD_POOL_CORE_SIZE_INCREMENT_TWOFOLD) {
            // 线程池空闲时，减少核心线程，但始终不少于初始线程
            threadPool.setCorePoolSize(threadPool.getCorePoolSize() - THREAD_POOL_CORE_SIZE_INCREMENT);
        }
    }


    public void processDecRemoteContext(DecRemoteContext<Serializable> context) {
        threadPool.submit(() -> {
            process(context);
        });
    }

    public void shutdown() {
        if(threadPool != null) {
            threadPool.shutdown();
        }
        if(null != coreSizeSchedule) {
            coreSizeSchedule.shutdown();
        }
    }

    private void process(DecRemoteContext<Serializable> context) {
        Object result = null;
        DecCloudApi cloudApi = context.getCloudDestination();
        Object bean = applicationContext.getBean(cloudApi.getBeanClass());
        try {
            Method method = cloudApi.getBeanClass().getMethod(cloudApi.getInterfaceMethod(), cloudApi.getParameterTypes());
            if(cloudApi.getParameterTypes().length > 0) {
                Object[] args = new Object[cloudApi.getParameterTypes().length];
                for (int i = 0; i < context.getParamList().size(); i++) {
                    args[i] = context.getParamList().get(i);
                }
                result = method.invoke(bean, args);
            } else {
                result = method.invoke(bean);
            }

            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS);
            context.setResult((Serializable) result);
        } catch (IllegalAccessException e) {
            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_FAILURE);
            context.setFailure(e);
            log.error("dec cloud outbound interface must be public", e);
        } catch (InvocationTargetException | NoSuchMethodException e) {
            context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_FAILURE);
            context.setFailure(e);
            log.error("invoke dec cloud outbound interface error", e);
        }

        // 远程调用成功, 返回结果
        context.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_2);
        DecCloudInvoker invoker = applicationContext.getBean(DecCloudInvoker.class);
        invoker.back(context);
    }

}
