package io.gitee.declear.dec.cloud.common.remoting.invoke;

import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.exception.DecCloudConnectException;
import io.gitee.declear.dec.cloud.common.remoting.DecPromise;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContext;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContextManager;
import io.gitee.declear.dec.cloud.common.rpc.protocol.DecCloudProtocol;
import io.gitee.declear.dec.cloud.common.rpc.protocol.pack.DecCloudProtocolPack;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * DecCloud 服务间调用工具 Invoker
 * @author DEC
 */
public class DecCloudInvoker {

    @Autowired
    private DecCloudProtocol decCloudProtocol;

    @Autowired
    private DecRemoteContextManager decRemoteContextManager;

    private static final Integer DEFAULT_RETRY_TIMES = 3;

    /**
     * 远程调用
     * @param context
     */
    public void invoke(DecRemoteContext<Serializable> context) {
        invoke(context, null, null);
    }

    /**
     * 远程调用
     * @param context
     * @param promiseSuccess
     * @param promiseError
     */
    public void invoke(DecRemoteContext<Serializable> context, DecPromise<DecRemoteContext<Serializable>> promiseSuccess, DecPromise<DecRemoteContext<Serializable>> promiseError) {
        try {
            decRemoteContextManager.putHoldRemoteContext(context);
            DecCloudProtocolPack pack = new DecCloudProtocolPack();
            pack.setType(context.getType());
            pack.setVersion(Constants.DEC_CLOUD_CODE_PROTOCOL_VERSION);
            pack.setId(context.getId());
            pack.setCloudOrigin(context.getCloudOrigin());
            pack.setCloudDestination(context.getCloudDestination());
            pack.setDataList(context.getParamList());
            decCloudProtocol.sendData(context, pack);
            if(null != promiseSuccess) {
                promiseSuccess.exec(context);
            }
        } catch (InterruptedException | RuntimeException e) {
            // 重试3次
            if(null == context.getRetryTimes()) {
                context.setRetryTimes(0);
            } else {
                context.setRetryTimes(context.getRetryTimes() + 1);
            }
            if(context.getRetryTimes() < DEFAULT_RETRY_TIMES) {
                invoke(context, promiseSuccess, promiseError);
            } else {
                // 重试失败， 记录日志并发送至 index
                decRemoteContextManager.pushException2Index(new DecCloudConnectException(String.format("%s send data to %s, retry over times.", context.getCloudOrigin(), context.getCloudDestination())));
                if(null != promiseError) {
                    promiseError.exec(context);
                }
            }
        }
    }

    /**
     * 远程调用返回结果
     * @param context
     */
    public void back(DecRemoteContext<Serializable> context) {
        try {
            DecCloudProtocolPack pack = new DecCloudProtocolPack();
            pack.setType(context.getType());
            pack.setVersion(Constants.DEC_CLOUD_CODE_PROTOCOL_VERSION);
            pack.setId(context.getId());
            List<Serializable> dataList = new ArrayList<>(2);
            dataList.add(context.getBackStatus());
            if(Objects.equals(context.getBackStatus(), DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS)) {
                dataList.add(context.getResult());
            } else {
                dataList.add(context.getFailure());
            }
            pack.setDataList(dataList);
            decCloudProtocol.sendResult(context, pack);
        } catch (InterruptedException | RuntimeException e) {
            // 重试3次
            if(null == context.getRetryTimes()) {
                context.setRetryTimes(0);
            } else {
                context.setRetryTimes(context.getRetryTimes() + 1);
            }
            if(context.getRetryTimes() < DEFAULT_RETRY_TIMES) {
                back(context);
            } else {
                // 重试失败， 记录日志并发送至 index
                decRemoteContextManager.pushException2Index(new DecCloudConnectException(String.format("%s send back result to %s, retry over times.", context.getCloudDestination(), context.getCloudOrigin())));
            }
        }
    }

}
