package io.gitee.declear.dec.cloud.common.remoting.resource;

import io.gitee.declear.common.utils.CommonUtils;
import io.gitee.declear.dec.cloud.common.annotation.DecCloudService;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.property.PropertiesManager;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.util.DigestUtils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Dec cloud框架相关的所有资源的管理类
 * @author DEC
 */
@Data
@Slf4j
public class DecCloudResourceManager implements ApplicationContextAware {

    @Autowired
    private PropertiesManager propertiesManager;

    private ApplicationContext applicationContext;

    /**
     * index 实例map
     */
    private Map<String, DecCloudResource> indexMap;
    private String indexMd5;

    /**
     * service/gateway 所属的主 index 实例, 只对 service/gateway 生效, index 实例无需此属性
     * 首先各 index 均分所有的 service 和 gateway
     * 之后根据每次 recruit 的时长再平衡
     * main index 失效后重新分配
     */
    private DecCloudResource mainIndex;

    /**
     * index 上一次 recruit 的持续时间
     */
    private Long recruitInterval;

    /**
     * gateway 实例map
     */
    private Map<String, DecCloudResource> gatewayMap;
    private String gatewayMd5;

    /**
     * service 实例map
     */
    private Map<String, DecCloudResource> serviceMap;
    private String serviceMd5;

    /**
     * service instance分组 map, 方便查找
     */
    private Map<String, Map<InetSocketAddress, DecCloudResource>> serviceInstanceMap;

    /**
     * 当前服务对外提供的接口集合
     */
    private Map<String, DecCloudApi> outBoundInterfaceMap;
    private String outBoundInterfaceMd5;

    /**
     * 当前服务引入的外部接口集合
     */
    private Map<String, DecCloudApi> inBoundInterfaceMap;

    /**
     * 通过DecCloudApi查找该接口所在的服务实例，获取ip等信息, 方便查找
     */
    private Map<String, List<DecCloudApi>> cloudApiMap;

    /**
     * 本服务实例
     */
    private DecCloudApi cloudOrigin;

    /**
     * 负载均衡处理策略处理器
     */
    private DecCloudLoadBalanceProcessor decCloudLoadBalanceProcessor;

    /**
     * 服务间通信处理器
     */
    private RecruitProcessor recruitProcessor;

    /**
     * cloud type: index / service /gateway
     */
    private String cloudType;

    public void init() {
        indexMap = new ConcurrentHashMap<>(2 ^ 3);
        gatewayMap = new ConcurrentHashMap<>(2 ^ 5);
        serviceMap = new ConcurrentHashMap<>(2 ^ 8);

        serviceInstanceMap = new ConcurrentHashMap<>(2 ^ 8);

        outBoundInterfaceMap = new ConcurrentHashMap<>(2 ^ 8);
        inBoundInterfaceMap = new ConcurrentHashMap<>(2 ^ 8);

        cloudApiMap = new ConcurrentHashMap<>(2 ^ 8);

        try {
            String port = propertiesManager.getProperty(Constants.DEC_CLOUD_PORT);
            String instance = propertiesManager.getProperty(Constants.DEC_CLOUD_INSTANCE_NAME);
            InetSocketAddress localAddress = new InetSocketAddress(CommonUtils.getLocalHostAddress().getHostAddress(), Integer.parseInt(port));
            cloudOrigin = new DecCloudApi();
            cloudOrigin.setAddress(localAddress);
            cloudOrigin.setInstance(instance);
        } catch (IOException e) {
            log.error("get cloudOrigin error", e);
        }

        recruitProcessor = new RecruitProcessor(DecCloudResourceManager.this, applicationContext);
        cloudType = propertiesManager.getProperty(Constants.DEC_CLOUD_TYPE);
    }

    public void shutdown() {
        if(null != recruitProcessor) {
            recruitProcessor.shutdown();
        }
    }

    public DecCloudApi getDecCloudApi(Method method) {
        return loadBalance(cloudApiMap.get(getDecCloudApiKey(method)));
    }

    private String getDecCloudApiKey(Method method) {
        DecCloudService decCloudService = method.getDeclaringClass().getAnnotation(DecCloudService.class);

        DecCloudApi cloudApi = new DecCloudApi();
        cloudApi.setInstance(decCloudService.instance());
        cloudApi.setInterfaceMethod(method.getName());
        cloudApi.setParameterTypes(method.getParameterTypes());
        return cloudApi.generateIdWithInstanceAndMethod();
    }

    public void putDecOutBoundInterfaces(Map<String, DecCloudApi> decOutBoundInterfaceMap) {
        this.outBoundInterfaceMap.putAll(decOutBoundInterfaceMap);
        outBoundInterfaceMd5 = generateMd5(this.outBoundInterfaceMap);
    }

    public void putDecInBoundInterfaces(Map<String, DecCloudApi> decInBoundInterfaceMap) {
        this.inBoundInterfaceMap.putAll(decInBoundInterfaceMap);
    }

    public void putIndexCloudResource(DecCloudResource index) {
        index.setIsActive(true);
        indexMap.put(index.generateId(), index);
        indexMd5 = generateMd5(indexMap);
    }

    public void putGatewayCloudResource(DecCloudResource gateway) {
        gateway.setIsActive(true);
        gatewayMap.put(gateway.generateId(), gateway);
        gatewayMd5 = generateMd5(gatewayMap);
    }

    public void putServiceCloudResource(DecCloudResource service) {
        service.setIsActive(true);
        serviceMap.put(service.generateId(), service);
        serviceMd5 = generateMd5(serviceMap);
        if(serviceInstanceMap.get(service.getInstance()) == null) {
            Map<InetSocketAddress, DecCloudResource> cloudResourceMap = new ConcurrentHashMap<>(100);
            cloudResourceMap.put(service.getAddress(), service);
            serviceInstanceMap.put(service.getInstance(), cloudResourceMap);
        } else {
            Map<InetSocketAddress, DecCloudResource> cloudResourceMap = serviceInstanceMap.get(service.getInstance());
            DecCloudResource cloudResource = cloudResourceMap.get(service.getAddress());
            if(cloudResource != null) {
                cloudResource.setIsActive(true);
                if(cloudResource.getDecCloudApiMap() != null) {
                    cloudResource.getDecCloudApiMap().putAll(service.getDecCloudApiMap());
                } else {
                    cloudResource.setDecCloudApiMap(service.getDecCloudApiMap());
                }
            } else {
                cloudResourceMap.put(service.getAddress(), service);
            }
        }
    }

    public void putServiceCloudApi(DecCloudApi cloudApi) {
        Map<InetSocketAddress, DecCloudResource> cloudResourceMap = serviceInstanceMap.get(cloudApi.getInstance());
        DecCloudResource cloudResource = cloudResourceMap.get(cloudApi.getAddress());
        if(cloudResource != null) {
            if(cloudResource.getDecCloudApiMap() != null) {
                cloudResource.getDecCloudApiMap().put(cloudApi.generateIdWithAddress(), cloudApi);
            } else {
                Map<String, DecCloudApi> decCloudApiMap = new ConcurrentHashMap<>(128);
                decCloudApiMap.put(cloudApi.generateIdWithAddress(), cloudApi);
                cloudResource.setDecCloudApiMap(decCloudApiMap);
            }
        } else {
            Map<String, DecCloudApi> decCloudApiMap = new ConcurrentHashMap<>(128);
            decCloudApiMap.put(cloudApi.generateIdWithAddress(), cloudApi);

            cloudResource = new DecCloudResource();
            cloudResource.setAddress(cloudApi.getAddress());
            cloudResource.setInstance(cloudApi.getInstance());
            cloudResource.setDecCloudApiMap(decCloudApiMap);

            cloudResourceMap.put(cloudApi.getAddress(), cloudResource);
        }

        putServiceCloudApiWithInBound(cloudApi);
    }

    /**
     * 将从index server过去到的其他服务的对外接口跟本服务的 @DecCloudService#@DecCloudInBound 进行匹配
     * @param cloudApi
     */
    private void putServiceCloudApiWithInBound(DecCloudApi cloudApi) {
        String key = cloudApi.generateIdWithInstanceAndMethod();
        if(inBoundInterfaceMap.containsKey(key)) {
            if (CommonUtils.isNotEmpty(cloudApiMap.get(key))) {
                cloudApiMap.get(key).add(cloudApi);
            } else {
                List<DecCloudApi> cloudApiList = new ArrayList<>();
                cloudApiList.add(cloudApi);
                cloudApiMap.put(key, cloudApiList);
            }
        }
    }

    private DecCloudApi loadBalance(List<DecCloudApi> cloudApiList) {
        if(CommonUtils.isNotEmpty(cloudApiList)) {
            if(cloudApiList.size() == 1) {
                return cloudApiList.get(0);
            }
        }

        return decCloudLoadBalanceProcessor.loadBalance(cloudApiList);
    }

    public <T> String generateMd5(Map<String, T> resourceMap) {
        if(CommonUtils.isNotEmpty(resourceMap)) {
            List<String> resourceIdList =new ArrayList<>();
            try {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                ObjectOutputStream oos = new ObjectOutputStream(bos);
                resourceMap.forEach((key, value) -> {
                    resourceIdList.add(key);
                });
                Collections.sort(resourceIdList);
                for(String resourceId : resourceIdList) {
                    oos.writeObject(resourceId);
                }

                return DigestUtils.md5DigestAsHex(bos.toByteArray());
            } catch (IOException e) {
                log.error("generate md5 string error.", e);
            }
        }
        return null;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
