package io.gitee.declear.dec.cloud.common.remoting.resource;

import io.gitee.declear.common.utils.CommonUtils;
import io.gitee.declear.common.utils.NamedThreadFactory;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContext;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContextManager;
import io.gitee.declear.dec.cloud.common.remoting.invoke.DecCloudInvoker;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;

import java.io.IOException;
import java.io.Serializable;
import java.net.InetSocketAddress;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * dec cloud个服务间建立联系的过程管理
 * 1 index 需要和其他 index 建立联系
 *   index 之间需要共享所拥有的的 index/service/gateway 资源列表
 *   index 监控 gateway/service 的心跳，将下线服务从资源列表中标记， 之后删除
 *   index 从 service/gateway 获取未在当前资源列表的 index
 *
 *
 * 2 gateway/service 需要从 index 获取 index/service/gateway 资源列表(同步)
 * @author DEC
 */
@Slf4j
public class RecruitProcessor {

    /**
     * 管理 从 index 同步资源列表的任务 的定时任务
     */
    private ScheduledExecutorService indexRecruitSchedule;

    /**
     * 管理 从 service 同步资源列表的任务 的定时任务
     */
    private ScheduledExecutorService serviceRecruitSchedule;

    /**
     * 管理 从 gateway 同步资源列表的任务 的定时任务
     */
    private ScheduledExecutorService gatewayRecruitSchedule;

    private DecCloudResourceManager decCloudResourceManager;

    private DecRemoteContextManager decRemoteContextManager;

    private ApplicationContext applicationContext;

    private static final String RECRUIT_TYPE_I_1 = "i1";
    private static final String RECRUIT_TYPE_I_2 = "i2";
    private static final String RECRUIT_TYPE_I_3 = "i3";
    private static final String RECRUIT_TYPE_I_4 = "i4";
    private static final String RECRUIT_TYPE_I_5 = "i5";
    private static final String RECRUIT_TYPE_I_6 = "i6";
    private static final String RECRUIT_TYPE_I_7 = "i7";
    private static final String RECRUIT_TYPE_I_8 = "i8";
    private static final String RECRUIT_TYPE_I_9 = "i9";
    private static final String RECRUIT_TYPE_I_10 = "i10";

    private static final String RECRUIT_TYPE_S_1 = "s1";
    private static final String RECRUIT_TYPE_S_2 = "s2";
    private static final String RECRUIT_TYPE_S_3 = "s3";
    private static final String RECRUIT_TYPE_S_4 = "s4";
    private static final String RECRUIT_TYPE_S_5 = "s5";

    private static final String RECRUIT_TYPE_API_1 = "api1";
    private static final String RECRUIT_TYPE_API_2 = "api2";
    private static final String RECRUIT_TYPE_API_3 = "api3";
    private static final String RECRUIT_TYPE_API_4 = "api4";
    private static final String RECRUIT_TYPE_API_5 = "api5";

    private static final Long RECRUIT_INTERVAL_CORDON = 1500L;

    public RecruitProcessor(DecCloudResourceManager decCloudResourceManager, ApplicationContext applicationContext) {
        this.decCloudResourceManager = decCloudResourceManager;
        this.applicationContext = applicationContext;
        init();
    }

    public void init() {
        decRemoteContextManager = applicationContext.getBean(DecRemoteContextManager.class);

        if(Objects.equals(decCloudResourceManager.getCloudType(), Constants.DEC_CLOUD_TYPE_INDEX)) {
            indexRecruitSchedule = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("schedule-pool-recruit-i"));
            indexRecruitSchedule.scheduleWithFixedDelay(() -> syncResourceListFromIndex(), 60, 30, TimeUnit.SECONDS);
        }

        if(Objects.equals(decCloudResourceManager.getCloudType(), Constants.DEC_CLOUD_TYPE_SERVICE)) {
            serviceRecruitSchedule = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("schedule-pool-recruit-s"));
            serviceRecruitSchedule.scheduleWithFixedDelay(() -> syncResourceListFromService(), 60, 30, TimeUnit.SECONDS);
        }

        if(Objects.equals(decCloudResourceManager.getCloudType(), Constants.DEC_CLOUD_TYPE_GATEWAY)) {
            gatewayRecruitSchedule = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("schedule-pool-recruit-g"));
            gatewayRecruitSchedule.scheduleWithFixedDelay(() -> syncResourceListFromGateway(), 60, 30, TimeUnit.SECONDS);
        }
    }

    public void shutdown() {
        if(null != indexRecruitSchedule) {
            indexRecruitSchedule.shutdown();
        }
        if(null != serviceRecruitSchedule) {
            serviceRecruitSchedule.shutdown();
        }
        if(null != gatewayRecruitSchedule) {
            gatewayRecruitSchedule.shutdown();
        }
    }

    private void syncResourceListFromIndex() {
        long startTime = System.currentTimeMillis();

        // Recruit indexMap
        processRecruitIndexMapStart();

        // Recruit serviceMap
        processRecruitServiceMapStart();

        // Recruit serviceMap 中每一个 service 的 decCloudApiMap
        processRecruitServiceApiStart();

        // 根据 index 的每次 Recruit 的时间， 再平衡并更新 service/gateway 的 main index
        decCloudResourceManager.setRecruitInterval(System.currentTimeMillis() - startTime);
        processReBalanceRecruitMainIndexStart();
    }

    private void syncResourceListFromService() {
        // 在 DecCloudResourceManager.indexMap 中任意选择一个 index 服务为 主index
        if(CommonUtils.isEmpty(decCloudResourceManager.getMainIndex())) {
            Map.Entry<String, DecCloudResource> entry = decCloudResourceManager.getIndexMap().entrySet().iterator().next();
            decCloudResourceManager.setMainIndex(entry.getValue());
        }
        // Recruit 主index 的 indexMap
        processRecruitIndexIndexMapStart();

        // Recruit 主index 的 serviceMap
        processRecruitIndexServiceMapStart();

        // Recruit serviceMap 中每一个 service 的 decCloudApiMap
        processRecruitServiceApiStart();
    }

    private void syncResourceListFromGateway() {
        // 在 DecCloudResourceManager.indexMap 中任意选择一个 index 服务为 主index
        if(CommonUtils.isEmpty(decCloudResourceManager.getMainIndex())) {
            Map.Entry<String, DecCloudResource> entry = decCloudResourceManager.getIndexMap().entrySet().iterator().next();
            decCloudResourceManager.setMainIndex(entry.getValue());
        }
        // Recruit 主index 的 indexMap
        processRecruitIndexIndexMapStart();

        // Recruit 主index 的 serviceMap
        processRecruitIndexServiceMapStart();

        // Recruit serviceMap 中每一个 service 的 decCloudApiMap
        processRecruitServiceApiStart();
    }

    public void process(DecRemoteContext<Serializable> context) {
        String recruitType = context.getParamList().get(0).toString();
        switch (recruitType) {
            case RECRUIT_TYPE_I_1 -> processRecruitTypeI1(context);
            case RECRUIT_TYPE_I_4 -> processRecruitTypeI4(context);
            case RECRUIT_TYPE_I_6 -> processRecruitTypeI6(context);
            case RECRUIT_TYPE_I_8 -> processRecruitTypeI8(context);
            case RECRUIT_TYPE_I_9 -> processRecruitTypeI9(context);
            case RECRUIT_TYPE_S_1 -> processRecruitTypeS1(context);
            case RECRUIT_TYPE_S_4 -> processRecruitTypeS4(context);
            case RECRUIT_TYPE_API_1 -> processRecruitTypeApi1(context);
            case RECRUIT_TYPE_API_4 -> processRecruitTypeApi4(context);
            default -> throw new IllegalStateException("recruit process unsupport recruit type: " + recruitType);
        }
    }

    public void back(DecRemoteContext<Serializable> context) {
        String recruitType = context.getResult().toString();
        switch (recruitType) {
            case RECRUIT_TYPE_I_2 -> processRecruitTypeI2(context);
            case RECRUIT_TYPE_I_3 -> processRecruitTypeI3(context);
            case RECRUIT_TYPE_I_5 -> processRecruitTypeI5(context);
            case RECRUIT_TYPE_I_7 -> processRecruitTypeI7(context);
            case RECRUIT_TYPE_I_10 -> processRecruitTypeI10(context);
            case RECRUIT_TYPE_S_2 -> processRecruitTypeS2(context);
            case RECRUIT_TYPE_S_3 -> processRecruitTypeS3(context);
            case RECRUIT_TYPE_S_5 -> processRecruitTypeS5(context);
            case RECRUIT_TYPE_API_2 -> processRecruitTypeApi2(context);
            case RECRUIT_TYPE_API_3 -> processRecruitTypeApi3(context);
            case RECRUIT_TYPE_API_5 -> processRecruitTypeApi5(context);
            default -> throw new IllegalStateException("recruit back unsupport recruit type: " + recruitType);
        }
    }

    private void processRecruitIndexMapStart() {
        if(CommonUtils.isNotEmpty(decCloudResourceManager.getIndexMap())) {
            decCloudResourceManager.getIndexMap().forEach((key, index) -> {
                DecCloudApi cloudDestination = new DecCloudApi();
                cloudDestination.setAddress(index.getAddress());
                cloudDestination.setInstance(index.getInstance());

                List<Serializable> paramList = new ArrayList<>();
                paramList.add(RECRUIT_TYPE_I_1);
                paramList.add(decCloudResourceManager.getIndexMd5());

                // 比对index节点index列表 MD5 值
                sendRecruitData(cloudDestination, paramList);
            });
        }
    }

    private void processRecruitIndexIndexMapStart() {
        DecCloudApi cloudDestination = new DecCloudApi();
        cloudDestination.setAddress(decCloudResourceManager.getMainIndex().getAddress());
        cloudDestination.setInstance(decCloudResourceManager.getMainIndex().getInstance());

        List<Serializable> paramList = new ArrayList<>();
        paramList.add(RECRUIT_TYPE_I_1);
        paramList.add(decCloudResourceManager.getIndexMd5());

        // 比对index节点index列表 MD5 值
        sendRecruitData(cloudDestination, paramList);
    }

    private void processRecruitServiceMapStart() {
        if(CommonUtils.isNotEmpty(decCloudResourceManager.getIndexMap())) {
            decCloudResourceManager.getIndexMap().forEach((key, index) -> {
                DecCloudApi cloudDestination = new DecCloudApi();
                cloudDestination.setAddress(index.getAddress());
                cloudDestination.setInstance(index.getInstance());

                List<Serializable> paramList = new ArrayList<>();
                paramList.add(RECRUIT_TYPE_S_1);
                paramList.add(decCloudResourceManager.getServiceMd5());

                // 比对index节点service列表 MD5 值
                sendRecruitData(cloudDestination, paramList);
            });
        }
    }

    private void processRecruitIndexServiceMapStart() {
        DecCloudApi cloudDestination = new DecCloudApi();
        cloudDestination.setAddress(decCloudResourceManager.getMainIndex().getAddress());
        cloudDestination.setInstance(decCloudResourceManager.getMainIndex().getInstance());

        List<Serializable> paramList = new ArrayList<>();
        paramList.add(RECRUIT_TYPE_S_1);
        paramList.add(decCloudResourceManager.getServiceMd5());

        // 比对index节点service列表 MD5 值
        sendRecruitData(cloudDestination, paramList);
    }

    private void processRecruitServiceApiStart() {
        if(CommonUtils.isNotEmpty(decCloudResourceManager.getServiceMap())) {
            decCloudResourceManager.getServiceMap().forEach((key, service) -> {
                DecCloudApi cloudDestination = new DecCloudApi();
                cloudDestination.setAddress(service.getAddress());
                cloudDestination.setInstance(service.getInstance());

                List<Serializable> paramList = new ArrayList<>();
                paramList.add(RECRUIT_TYPE_API_1);
                try {
                    paramList.add(service.generateIdApiMd5());
                } catch (IOException e) {
                    log.error("generate service api md5 string error.", e);
                }

                // 比对index节点service api列表 MD5 值
                sendRecruitData(cloudDestination, paramList);
            });
        }
    }

    private void processReBalanceRecruitMainIndexStart() {
        if(CommonUtils.isNotEmpty(decCloudResourceManager.getIndexMap())) {
            decCloudResourceManager.getIndexMap().forEach((key, index) -> {
                DecCloudApi cloudDestination = new DecCloudApi();
                cloudDestination.setAddress(index.getAddress());
                cloudDestination.setInstance(index.getInstance());

                List<Serializable> paramList = new ArrayList<>();
                paramList.add(RECRUIT_TYPE_I_8);
                paramList.add(decCloudResourceManager.getRecruitInterval());

                // 发送 index 上一次 recruit 的持续时间
                sendRecruitData(cloudDestination, paramList);
            });
        }
    }

    private void sendRecruitData(DecCloudApi cloudDestination, List<Serializable> paramList) {
        DecCloudInvoker invoker = applicationContext.getBean(DecCloudInvoker.class);
        DecRemoteContext<Serializable> context = new DecRemoteContext<>();
        context.setId(CommonUtils.UUID());
        context.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_10);
        context.setCloudDestination(cloudDestination);
        context.setCloudOrigin(decCloudResourceManager.getCloudOrigin());
        context.setParamList(paramList);

        invoker.invoke(context);
    }

    private void sendRecruitBackData(DecRemoteContext<Serializable> context, Serializable result) {
        DecCloudInvoker invoker = applicationContext.getBean(DecCloudInvoker.class);
        context.setType(Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_11);
        context.setBackStatus(DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS);
        context.setResult(result);

        invoker.back(context);
    }

    private void processRecruitTypeI1(DecRemoteContext<Serializable> context) {
        String md5 = context.getParamList().get(1).toString();
        if(Objects.equals(md5, decCloudResourceManager.getIndexMd5())) {
            // 比对index节点index列表 MD5 值, 相同
            sendRecruitBackData(context, RECRUIT_TYPE_I_2);
        } else {
            // 比对index节点index列表 MD5 值, 不相同
            sendRecruitBackData(context, RECRUIT_TYPE_I_3);

            // 发送index节点列表
            List<Serializable> paramList = new ArrayList<>();
            paramList.add(RECRUIT_TYPE_I_4);
            decCloudResourceManager.getIndexMap().forEach((id, index) -> {
                if(Objects.equals(index.getIsActive(), true)) {
                    paramList.add(index.generateId());
                }
            });
            sendRecruitData(context.getCloudOrigin(), paramList);
        }
    }

    private void processRecruitTypeI2(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) md5 string is same as index ({}), did not do anything.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeI3(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) md5 string is not same as index ({}), synchronizing start.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeI4(DecRemoteContext<Serializable> context) {
        Map<String, DecCloudResource> indexMap = new HashMap<>(2 ^ 3);
        indexMap.putAll(decCloudResourceManager.getIndexMap());
        for (int i = 1; i < context.getParamList().size(); i++) {
            String indexId = context.getParamList().get(i).toString();
            if(null == decCloudResourceManager.getIndexMap().get(indexId)) {
                DecCloudResource index = new DecCloudResource();
                String[] address = indexId.substring(0, indexId.indexOf("/")).split(":");
                index.setAddress(new InetSocketAddress(address[0], Integer.parseInt(address[1])));
                index.setInstance(indexId.substring(indexId.indexOf("/") + 1));
                decCloudResourceManager.putIndexCloudResource(index);
            }
            indexMap.remove(indexId);
        }

        sendRecruitBackData(context, RECRUIT_TYPE_I_5);

        // 上报 主index 的 indexMap 中没有记录但本服务中包含的 index
        if(Objects.equals(decCloudResourceManager.getCloudType(), Constants.DEC_CLOUD_TYPE_SERVICE)
                || Objects.equals(decCloudResourceManager.getCloudType(), Constants.DEC_CLOUD_TYPE_GATEWAY)) {
            if(CommonUtils.isNotEmpty(indexMap)) {
                List<Serializable> paramList = new ArrayList<>();
                paramList.add(RECRUIT_TYPE_I_6);
                indexMap.forEach((id, index) -> {
                    if(Objects.equals(index.getIsActive(), true)) {
                        paramList.add(id);
                    }
                });
                sendRecruitData(context.getCloudOrigin(), paramList);
            }
        }
    }

    private void processRecruitTypeI5(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) md5 string is not same as index ({}), synchronizing end.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeI6(DecRemoteContext<Serializable> context) {
        for (int i = 1; i < context.getParamList().size(); i++) {
            String indexId = context.getParamList().get(i).toString();
            if(null == decCloudResourceManager.getIndexMap().get(indexId)) {
                DecCloudResource index = new DecCloudResource();
                String[] address = indexId.substring(0, indexId.indexOf("/")).split(":");
                index.setAddress(new InetSocketAddress(address[0], Integer.parseInt(address[1])));
                index.setInstance(indexId.substring(indexId.indexOf("/") + 1));
                decCloudResourceManager.putIndexCloudResource(index);
            }
        }

        sendRecruitBackData(context, RECRUIT_TYPE_I_7);
    }

    private void processRecruitTypeI7(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        for (int i = 1; i < originContext.getParamList().size(); i++) {
            log.info("recruit: report index ({}) to index({}) success.", originContext.getParamList().get(i), originContext.getCloudDestination().generateIdWithInstance());
        }

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeI8(DecRemoteContext<Serializable> context) {
        if(CommonUtils.isNotEmpty(decCloudResourceManager.getRecruitInterval())) {
            Long interval = (Long) context.getParamList().get(1);
            if(interval < decCloudResourceManager.getRecruitInterval() - RECRUIT_INTERVAL_CORDON) {
                // 选择一个 service/gateway 再平衡 main index
                List<Serializable> paramList = new ArrayList<>();
                paramList.add(RECRUIT_TYPE_I_9);
                paramList.add(context.getCloudOrigin().generateIdWithInstance());

                sendRecruitData(context.getCloudOrigin(), paramList);
            }
        }

        sendRecruitBackData(context, RECRUIT_TYPE_I_10);
    }

    private void processRecruitTypeI9(DecRemoteContext<Serializable> context) {
        String indexId = context.getParamList().get(1).toString();
        if(null != decCloudResourceManager.getIndexMap().get(indexId)) {
            decCloudResourceManager.setMainIndex(decCloudResourceManager.getIndexMap().get(indexId));
        }
        log.info("recruit: ({}) rebalance main index ({}) success.", context.getCloudDestination().generateIdWithInstance(), context.getParamList().get(1));
    }

    private void processRecruitTypeI10(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: rebalance with index ({}) success.", originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeS1(DecRemoteContext<Serializable> context) {
        String md5 = context.getParamList().get(1).toString();
        if(Objects.equals(md5, decCloudResourceManager.getServiceMd5())) {
            // 比对index节点service列表 MD5 值, 相同
            sendRecruitBackData(context, RECRUIT_TYPE_S_2);
        } else {
            // 比对index节点service列表 MD5 值, 不相同
            sendRecruitBackData(context, RECRUIT_TYPE_S_3);

            // 发送index节点列表
            List<Serializable> paramList = new ArrayList<>();
            paramList.add(RECRUIT_TYPE_S_4);
            decCloudResourceManager.getServiceMap().forEach((id, service) -> {
                if(Objects.equals(service.getIsActive(), true)) {
                    paramList.add(service.generateId());
                }
            });
            sendRecruitData(context.getCloudOrigin(), paramList);
        }
    }

    private void processRecruitTypeS2(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) service md5 string is same as index ({}), did not do anything.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeS3(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) service md5 string is not same as index ({}), synchronizing start.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeS4(DecRemoteContext<Serializable> context) {
        for (int i = 1; i < context.getParamList().size(); i++) {
            String serviceId = context.getParamList().get(i).toString();
            if(null == decCloudResourceManager.getServiceMap().get(serviceId)) {
                DecCloudResource service = new DecCloudResource();
                String[] address = serviceId.substring(0, serviceId.indexOf("/")).split(":");
                service.setAddress(new InetSocketAddress(address[0], Integer.parseInt(address[1])));
                service.setInstance(serviceId.substring(serviceId.indexOf("/") + 1));
                decCloudResourceManager.putIndexCloudResource(service);
            }
        }

        sendRecruitBackData(context, RECRUIT_TYPE_S_5);
    }

    private void processRecruitTypeS5(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) md5 string is not same as index ({}), synchronizing end.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeApi1(DecRemoteContext<Serializable> context) {
        String md5 = context.getParamList().get(1).toString();
        if(Objects.equals(md5, decCloudResourceManager.getOutBoundInterfaceMd5())) {
            // 比对index节点index列表 MD5 值, 相同
            sendRecruitBackData(context, RECRUIT_TYPE_API_2);
        } else {
            // 比对index节点index列表 MD5 值, 不相同
            sendRecruitBackData(context, RECRUIT_TYPE_API_3);

            // 发送 API 列表
            List<Serializable> paramList = new ArrayList<>();
            paramList.add(RECRUIT_TYPE_API_4);
            decCloudResourceManager.getOutBoundInterfaceMap().forEach((id, api) -> paramList.add(api));

            sendRecruitData(context.getCloudOrigin(), paramList);
        }
    }

    private void processRecruitTypeApi2(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) service api md5 string is same as index ({}), did not do anything.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeApi3(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) service api md5 string is not same as index ({}), synchronizing start.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

    private void processRecruitTypeApi4(DecRemoteContext<Serializable> context) {
        if(context.getParamList().size() > 1) {
            DecCloudResource service = null;
            Map<String, DecCloudApi> decCloudApiMap = new ConcurrentHashMap<>(128);
            for (int i = 1; i < context.getParamList().size(); i++) {
                Serializable api = context.getParamList().get(i);
                if (api instanceof DecCloudApi cloudApi) {
                    if(null == service) {
                        service = decCloudResourceManager.getServiceMap().get(cloudApi.generateIdWithInstance());
                    }
                    decCloudApiMap.put(cloudApi.generateIdWithAddress(), cloudApi);
                }
            }

            if(null != service) {
                service.setDecCloudApiMap(decCloudApiMap);
            }
        }

        sendRecruitBackData(context, RECRUIT_TYPE_API_5);
    }

    private void processRecruitTypeApi5(DecRemoteContext<Serializable> context) {
        DecRemoteContext<Serializable> originContext = decRemoteContextManager.getHoldRemoteContext(context.getId());
        log.info("recruit: index ({}) service api md5 string is same as index ({}), synchronizing end.", originContext.getCloudOrigin().generateIdWithInstance(), originContext.getCloudDestination().generateIdWithInstance());

        decRemoteContextManager.removeHoldRemoteContext(context.getId());
    }

}
