package io.gitee.declear.dec.cloud.common.rpc.netty;

import io.gitee.declear.common.utils.CommonUtils;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.exception.ChannelConnectException;
import io.gitee.declear.dec.cloud.common.property.PropertiesManager;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContext;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContextManager;
import io.gitee.declear.dec.cloud.common.rpc.protocol.code.DecCloudCodeAdapter;
import io.gitee.declear.dec.cloud.common.rpc.protocol.code.DecCloudPackageCodeAdapter;
import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.channel.*;
import io.netty.channel.socket.SocketChannel;
import io.netty.handler.proxy.Socks5ProxyHandler;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.concurrent.Future;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.io.Serializable;
import java.net.InetSocketAddress;
import java.util.Objects;
import java.util.regex.Pattern;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

/**
 * Netty Client
 * @author DEC
 */
@Slf4j
public class NettyClient {

    private static final Pattern LOCAL_IP_PATTERN = Pattern.compile("127(\\.\\d{1,3}){3}$");
    private static final String LOCAL_HOST = "localhost";
    private static final String DEC_CLOUD_NETTY_SSL_ENABLED_DEFAULT_VALUE = "false";

    @Autowired
    private PropertiesManager propertiesManager;

    @Autowired
    private NettyGlobalResourceManager globalResourceManager;

    @Autowired
    private DecRemoteContextManager decRemoteContextManager;

    private Bootstrap bootstrap;

    private EventLoopGroup workerGroup;

    public void init() {
        globalResourceManager.setNettyClient(this);
        bootstrap = new Bootstrap();

        workerGroup = NettyEventLoopFactory.eventLoopGroup(Constants.DEFAULT_IO_THREADS, Constants.EVENT_LOOP_CLIENT_WORKER_POOL_NAME);

        bootstrap.group(workerGroup)
                .option(ChannelOption.SO_KEEPALIVE, Boolean.TRUE)
                .option(ChannelOption.TCP_NODELAY, Boolean.TRUE)
                .option(ChannelOption.ALLOCATOR, PooledByteBufAllocator.DEFAULT)
                .channel(NettyEventLoopFactory.socketChannelClass())
                .handler(new ChannelInitializer<SocketChannel>() {
                    @Override
                    protected void initChannel(SocketChannel ch) throws Exception {

                        DecCloudCodeAdapter decCloudCodeAdapter = new DecCloudCodeAdapter();
                        DecCloudPackageCodeAdapter decCloudPackageCodeAdapter = new DecCloudPackageCodeAdapter();

                        ch.pipeline()
                                .addLast("packEncoder", decCloudPackageCodeAdapter.getEncoder())
                                .addLast("decCloudEncoder", decCloudCodeAdapter.getEncoder())
                                .addLast("clientIdleHandler", new IdleStateHandler(Constants.NETTY_CLIENT_DEFAULT_HEARTBEAT, 0, 0, MILLISECONDS))
                                .addLast("handler", new NettyClientHandler(NettyClient.this));

                        boolean sslEnabled = Boolean.parseBoolean(propertiesManager.getProperty(Constants.DEC_CLOUD_NETTY_SSL_ENABLED_KEY, DEC_CLOUD_NETTY_SSL_ENABLED_DEFAULT_VALUE));
                        if(sslEnabled) {
                            ch.pipeline().addFirst(SslContextProvider.getClientSslContext(propertiesManager).newHandler(ch.alloc()));
                        }

                        String socksProxyHost = propertiesManager.getProperty(Constants.DEC_CLOUD_NETTY_CLIENT_SOCKS_PROXY_HOST_KEY);
                        if(socksProxyHost != null && !isLocalHost(socksProxyHost)) {
                            int socksProxyPort = Integer.parseInt(propertiesManager.getProperty(Constants.DEC_CLOUD_NETTY_CLIENT_SOCKS_PROXY_PORT_KEY));
                            Socks5ProxyHandler socks5ProxyHandler = new Socks5ProxyHandler(new InetSocketAddress(socksProxyHost, socksProxyPort));
                            ch.pipeline().addFirst(socks5ProxyHandler);
                        }
                    }
                });
        log.info("netty client init complete.");
    }

    public Channel connect(InetSocketAddress address) throws InterruptedException {
        ChannelFuture future = bootstrap.connect(address).sync();
        boolean ret = future.awaitUninterruptibly(Constants.NETTY_CLIENT_DEFAULT_CONNECT_TIMEOUT, MILLISECONDS);

        if (ret && future.isSuccess()) {
            Channel channel = future.channel();
            channel.closeFuture();
            globalResourceManager.addClientChannel(address, channel);
            return channel;
        } else {
            throw new ChannelConnectException(String.format("connect server: %s error.", address));
        }
    }

    public void disConnect(InetSocketAddress address) throws InterruptedException {
        Channel channel = globalResourceManager.getClientChannel(address);
        if(channel != null) {
            channel.close();
            globalResourceManager.removeClientChannel(address);
        }
    }

    public void shutdown() {
        try {
            long timeout = Constants.NETTY_CLIENT_DEFAULT_SHUTDOWN_TIMEOUT;
            long quietPeriod = Math.min(2000L, timeout);
            Future<?> workerGroupShutdownFuture = workerGroup.shutdownGracefully(quietPeriod, timeout, MILLISECONDS);
            workerGroupShutdownFuture.syncUninterruptibly();
        } catch (Throwable e) {
            log.error(e.getMessage(), e);
        }
    }

    public void receiveRemoteContext(DecRemoteContext<Serializable> context) {
        decRemoteContextManager.putProcessRemoteContext(context);
    }

    private Boolean isLocalHost(String host) throws IOException {
        return Objects.equals(CommonUtils.getLocalHostAddress().getHostAddress(), host)
                || LOCAL_IP_PATTERN.matcher(host).matches()
                || LOCAL_HOST.equalsIgnoreCase(host);
    }

}
