package io.gitee.declear.dec.cloud.common.rpc.netty;

import io.gitee.declear.common.utils.CommonUtils;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.property.PropertiesManager;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContext;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContextManager;
import io.gitee.declear.dec.cloud.common.rpc.protocol.code.DecCloudCodeAdapter;
import io.gitee.declear.dec.cloud.common.rpc.protocol.code.DecCloudHttpCodeAdapter;
import io.gitee.declear.dec.cloud.common.rpc.protocol.code.DecCloudPackageCodeAdapter;
import io.gitee.declear.dec.cloud.common.web.context.DecWebContext;
import io.gitee.declear.dec.cloud.common.web.context.DecWebContextManager;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.buffer.PooledByteBufAllocator;
import io.netty.channel.*;
import io.netty.channel.socket.SocketChannel;
import io.netty.handler.codec.http.HttpObjectAggregator;
import io.netty.handler.codec.http.HttpServerCodec;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.concurrent.Future;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.io.Serializable;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

/**
 * Netty Server
 * @author DEC
 */
@Slf4j
public class NettyServer {

    private static final String DEC_CLOUD_NETTY_SSL_ENABLED_DEFAULT_VALUE = "false";

    @Autowired
    private PropertiesManager propertiesManager;

    @Autowired
    private NettyGlobalResourceManager globalResourceManager;

    @Autowired
    private DecRemoteContextManager decRemoteContextManager;

    @Autowired
    private DecWebContextManager decWebContextManager;

    private ServerBootstrap bootstrap;

    private Channel channel;

    private EventLoopGroup bossGroup;
    private EventLoopGroup workerGroup;

    public void init() {
        globalResourceManager.setNettyServer(this);
        bootstrap = new ServerBootstrap();
        bossGroup = NettyEventLoopFactory.eventLoopGroup(1, Constants.EVENT_LOOP_SERVER_BOSS_POOL_NAME);
        workerGroup = NettyEventLoopFactory.eventLoopGroup(Constants.DEFAULT_IO_THREADS, Constants.EVENT_LOOP_SERVER_WORKER_POOL_NAME);

        bootstrap.group(bossGroup, workerGroup)
                .channel(NettyEventLoopFactory.serverSocketChannelClass())
                .option(ChannelOption.SO_REUSEADDR, Boolean.TRUE)
                .childOption(ChannelOption.TCP_NODELAY, Boolean.TRUE)
                .childOption(ChannelOption.SO_KEEPALIVE, Boolean.FALSE)
                .childOption(ChannelOption.ALLOCATOR, PooledByteBufAllocator.DEFAULT)
                .childHandler(new ChannelInitializer<SocketChannel>() {
                    @Override
                    protected void initChannel(SocketChannel ch) throws Exception {

                        DecCloudCodeAdapter decCloudCodeAdapter = new DecCloudCodeAdapter();
                        DecCloudPackageCodeAdapter decCloudPackageCodeAdapter = new DecCloudPackageCodeAdapter();
                        DecCloudHttpCodeAdapter decCloudHttpCodeAdapter = new DecCloudHttpCodeAdapter();

                        int idleTimeout = Integer.parseInt(propertiesManager.getProperty(Constants.DEC_CLOUD_NETTY_CHANNEL_IDLE_TIMEOUT_KEY, String.valueOf(Constants.NETTY_CHANNEL_IDLE_TIMEOUT_DEFAULT_VALUE)));
                        ch.pipeline()
                                .addLast("packDecoder", decCloudPackageCodeAdapter.getDecoder())
                                .addLast("decCloudDecoder", decCloudCodeAdapter.getDecoder())
                                .addLast("httpServerCoder", new HttpServerCodec())
                                .addLast("decCloudHttpResponseEncoder", decCloudHttpCodeAdapter.getEncoder())
                                .addLast("httpAggregator", new HttpObjectAggregator(512 * 1024))
                                .addLast("serverIdleHandler", new IdleStateHandler(0, 0, idleTimeout, MILLISECONDS))
                                .addLast("handler", new NettyServerHandler(NettyServer.this));

                        boolean sslEnabled = Boolean.parseBoolean(propertiesManager.getProperty(Constants.DEC_CLOUD_NETTY_SSL_ENABLED_KEY, DEC_CLOUD_NETTY_SSL_ENABLED_DEFAULT_VALUE));
                        if(sslEnabled) {
                            ch.pipeline().addFirst(SslContextProvider.getServerSslContext(propertiesManager).newHandler(ch.alloc()));
                        }

                    }
                });
        log.info("netty server init complete.");
    }

    public void start(int port) throws InterruptedException, IOException {
        String bindIp = CommonUtils.getLocalHostAddress().getHostAddress();
        ChannelFuture channelFuture = bootstrap.bind(port).sync();
        log.info("netty server bind ip: {} port: {}.", bindIp, port);
        channel = channelFuture.channel();

        channel.closeFuture();
    }

    public void shutdown() {
        try {
            if (channel != null) {
                channel.close();
            }

            long timeout = Constants.NETTY_SERVER_DEFAULT_SHUTDOWN_TIMEOUT;
            long quietPeriod = Math.min(2000L, timeout);
            Future<?> bossGroupShutdownFuture = bossGroup.shutdownGracefully(quietPeriod, timeout, MILLISECONDS);
            Future<?> workerGroupShutdownFuture = workerGroup.shutdownGracefully(quietPeriod, timeout, MILLISECONDS);
            bossGroupShutdownFuture.syncUninterruptibly();
            workerGroupShutdownFuture.syncUninterruptibly();
        } catch (Throwable e) {
            log.error(e.getMessage(), e);
        }
        log.info("netty server shutdown.");
    }

    public void receiveRemoteContext(DecRemoteContext<Serializable> context) {
        decRemoteContextManager.putProcessRemoteContext(context);
    }

    public void receiveWebContext(DecWebContext context) {
        decWebContextManager.putProcessWebContext(context);
    }
}
