package io.gitee.declear.dec.cloud.common.rpc.netty;

import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.property.PropertiesManager;
import io.netty.handler.ssl.*;
import lombok.extern.slf4j.Slf4j;

import javax.net.ssl.SSLException;
import java.io.IOException;
import java.io.InputStream;
import java.security.Provider;
import java.security.Security;

/**
 * SslContext Provider
 * provide netty server and client ssl context
 * @author DEC
 */
@Slf4j
public class SslContextProvider {

    private static SslContext serverSslContext;

    private static SslContext clientSslContext;

    private SslContextProvider() {}

    public static SslContext getServerSslContext(PropertiesManager propertiesManager) {
        if(null == serverSslContext) {
            serverSslContext = buildServerSslContext(propertiesManager);
        }

        return serverSslContext;
    }

    public static SslContext getClientSslContext(PropertiesManager propertiesManager) {
        if(null == clientSslContext) {
            clientSslContext = buildClientSslContext(propertiesManager);
        }

        return clientSslContext;
    }

    private static SslContext buildServerSslContext(PropertiesManager propertiesManager) {
        SslContextBuilder sslClientContextBuilder;
        InputStream serverKeyCertChainPathStream = null;
        InputStream serverPrivateKeyPathStream = null;
        InputStream serverTrustCertCollectionPathStream = null;
        try {
            serverKeyCertChainPathStream = propertiesManager.getFilePropertyInputStream(Constants.DEC_CLOUD_NETTY_SSL_SERVER_KEY_CERT_CHAIN_PATH);
            serverPrivateKeyPathStream = propertiesManager.getFilePropertyInputStream(Constants.DEC_CLOUD_NETTY_SSL_SERVER_PRIVATE_KEY_PATH);
            serverTrustCertCollectionPathStream = propertiesManager.getFilePropertyInputStream(Constants.DEC_CLOUD_NETTY_SSL_SERVER_TRUST_CERT_COLLECTION_PATH);
            String password = propertiesManager.getProperty(Constants.DEC_CLOUD_NETTY_SSL_SERVER_KEY_PASSWORD);
            if (password != null) {
                sslClientContextBuilder = SslContextBuilder.forServer(serverKeyCertChainPathStream,
                        serverPrivateKeyPathStream, password);
            } else {
                sslClientContextBuilder = SslContextBuilder.forServer(serverKeyCertChainPathStream,
                        serverPrivateKeyPathStream);
            }

            if (serverTrustCertCollectionPathStream != null) {
                sslClientContextBuilder.trustManager(serverTrustCertCollectionPathStream);
                sslClientContextBuilder.clientAuth(ClientAuth.REQUIRE);
            }
        } catch (Exception e) {
            throw new IllegalArgumentException("Could not find certificate file or the certificate is invalid.", e);
        } finally {
            safeCloseStream(serverKeyCertChainPathStream);
            safeCloseStream(serverPrivateKeyPathStream);
            safeCloseStream(serverTrustCertCollectionPathStream);
        }
        try {
            return sslClientContextBuilder.sslProvider(findSslProvider()).build();
        } catch (SSLException e) {
            throw new IllegalStateException("Build SslSession failed.", e);
        }
    }

    private static SslContext buildClientSslContext(PropertiesManager propertiesManager) {
        SslContextBuilder builder = SslContextBuilder.forClient();
        InputStream clientKeyCertChainPathStream = null;
        InputStream clientPrivateKeyPathStream = null;
        InputStream clientTrustCertCollectionPathStream = null;
        try {
            clientKeyCertChainPathStream = propertiesManager.getFilePropertyInputStream(Constants.DEC_CLOUD_NETTY_SSL_CLIENT_KEY_CERT_CHAIN_PATH);
            clientPrivateKeyPathStream = propertiesManager.getFilePropertyInputStream(Constants.DEC_CLOUD_NETTY_SSL_CLIENT_PRIVATE_KEY_PATH);
            clientTrustCertCollectionPathStream = propertiesManager.getFilePropertyInputStream(Constants.DEC_CLOUD_NETTY_SSL_CLIENT_TRUST_CERT_COLLECTION_PATH);

            if (clientTrustCertCollectionPathStream != null) {
                builder.trustManager(clientTrustCertCollectionPathStream);
            }
            if (clientKeyCertChainPathStream != null && clientKeyCertChainPathStream != null) {
                String password = propertiesManager.getProperty(Constants.DEC_CLOUD_NETTY_SSL_CLIENT_KEY_PASSWORD);
                if (password != null) {
                    builder.keyManager(clientKeyCertChainPathStream, clientPrivateKeyPathStream, password);
                } else {
                    builder.keyManager(clientKeyCertChainPathStream, clientPrivateKeyPathStream);
                }
            }
        } catch (Exception e) {
            throw new IllegalArgumentException("Could not find certificate file or find invalid certificate.", e);
        } finally {
            safeCloseStream(clientKeyCertChainPathStream);
            safeCloseStream(clientPrivateKeyPathStream);
            safeCloseStream(clientTrustCertCollectionPathStream);
        }
        try {
            return builder.sslProvider(findSslProvider()).build();
        } catch (SSLException e) {
            throw new IllegalStateException("Build SslSession failed.", e);
        }
    }

    private static void safeCloseStream(InputStream stream) {
        if (stream == null) {
            return;
        }
        try {
            stream.close();
        } catch (IOException e) {
            log.warn("SslContextProvider failed to close a stream.", e);
        }
    }

    /**
     * Returns OpenSSL if available, otherwise returns the JDK provider.
     */
    private static SslProvider findSslProvider() {
        if (OpenSsl.isAvailable()) {
            log.debug(" SslContextProvider using OPENSSL provider.");
            return SslProvider.OPENSSL;
        }
        if (checkJdkProvider()) {
            log.debug("SslContextProvider using JDK provider.");
            return SslProvider.JDK;
        }
        throw new IllegalStateException(
                "SslContextProvider could not find any valid TLS provider, please check your dependency or deployment environment, " +
                        "usually netty-tcnative, Conscrypt, or Jetty NPN/ALPN is needed.");
    }

    private static boolean checkJdkProvider() {
        Provider[] jdkProviders = Security.getProviders("SSLContext.TLS");
        return (jdkProviders != null && jdkProviders.length > 0);
    }

}
